(* Tools *)
let assoc_ll thy thm ll = List.assoc thm (List.assoc thy ll);;
let assoc_lll pn thy thm lll = 
	(List.assoc thm (List.assoc thy (List.assoc pn lll)));;

(* Reading tools *)
let file_iter fname fn =
  let ic = try open_in fname with Sys_error _ -> failwith ("file_iter: "^fname) in
  let next = ref 0 in
  let rec suck_lines () = fn !next (input_line ic); incr next; suck_lines () in
  try suck_lines () with End_of_file -> close_in ic
;;
let rec all_subdirh acc dirh = 
	try 
	  let s = Unix.readdir dirh in
	    if s <> "." && s <> ".." 
		  then all_subdirh (s :: acc) dirh
	    else all_subdirh acc dirh
	with End_of_file -> acc
;;
let all_subdir dir =
  let dirh = Unix.opendir dir in 
  let ret = all_subdirh [] dirh in
		Unix.closedir dirh; ret
;;

(* Results *)
let reg1 = Str.regexp ("SZS status Theorem\\|: Valid\\|SPASS beiseite:" ^
                      " Proof found.\\|^THEOREM PROVED$");;
let reg2 = Str.regexp "SZS status CounterSatisfiable\\|Non-Theorem";;
let reg3 = Str.regexp ("SZS status Timeout\\|SZS status Unknown\\|: Unknown\\|" ^
  "SZS status ResourceOut\\|^SPASS beiseite: Ran out of time. SPASS was killed.$");;
let reg4 = Str.regexp "[eE]rror\\|HighFailure\\|ExitFailure\\|PARSE ERROR";;
let find_rxp r s = 
	try Str.search_forward r s 0 >= 0 with _ -> false
;;
let grep_result s = 
	if find_rxp reg1 s then "Theorem" 
	else if find_rxp reg2 s then "CounterSatisfiable"
  else if find_rxp reg3 s then "Unknown" 
	else if find_rxp reg4 s then "Failure"
  else "NotFound"
;;

let mk_ithyl idir = 
  let thyl = List.filter (fun x -> x <> "quantHeuristics") (all_subdir idir) in
    List.map (fun x -> (x,all_subdir (idir ^ "/" ^ x))) thyl
;;
(* Reading *)				
let read_stats_pn_thy_thm odir pn thy thm = 
	try
	let fname = odir ^ "/" ^ pn ^ "/" ^ thy ^ "/" ^ thm in
	let s = ref "" in
	  file_iter fname (fun _ x -> s := x ^ " " ^ (!s));  
    (thm,grep_result (!s))
  with _ -> (thm,"Cannot open")
;;
let read_stats_pn_thy odir pn (thy,thml) =
	(thy, List.map (read_stats_pn_thy_thm odir pn thy) thml)
;;		
let read_stats_pn odir ithyl pn =
	(pn, List.map (read_stats_pn_thy odir pn) ithyl)
;;	
let read_stats odir ithyl pnl =
	List.map (read_stats_pn odir ithyl) pnl
;;

(* Number tools *)
let rec sum l = match l with
    [] -> 0
  | a :: m -> a + sum m
;;
let rec count_string n s sl = match sl with
    [] ->  n
	| a :: m -> if s = a then count_string (n+1) s m else count_string n s m
;;
let nb_of_ithm ithyl thy = List.length (List.assoc thy ithyl);;
let nb_of_unique ulll pn thy = assoc_ll pn thy ulll;;
let nb_of_pthm lll pn thy = 
	count_string 0 "Theorem" (List.map snd (assoc_ll pn thy lll));;

(* Inversion *)
let proved_by lll thy thm pn = assoc_lll pn thy thm lll = "Theorem";;	

let inv_stats_thy_thm lll pnl thy thm =
	List.filter (proved_by lll thy thm) pnl
;;	  
let inv_stats_thy lll pnl (thy,thml) = 
	List.map (fun x -> (x,inv_stats_thy_thm lll pnl thy x)) thml
;;
let inv_stats lll pnl ithyl =
	List.map (fun (thy,thml) -> (thy,inv_stats_thy lll pnl (thy,thml))) ithyl
;;						

let is_unique pn (thm,pnl) = pnl = [pn];;
let is_proved (thm,pnl) = pnl <> [];;

let unique_stats_pn_thy pn (thy,il) = 
	(thy,List.length (List.filter (is_unique pn) il));;

let unique_stats_pn ill pn = (pn,List.map (unique_stats_pn_thy pn) ill);;
let unique_stats_pnl ill pnl = List.map (unique_stats_pn ill) pnl;;

let combine_stats_thy (thy,il) = (thy,List.length (List.filter (is_proved) il));;
let combine_stats ill = List.map (combine_stats_thy) ill;;

(* Output tools *)
let oc = ref stdout;;
let os s = output_string !oc s;;
let rec oiter oc sep f = function
  [] -> () | [e] -> f e | h :: t -> f h; output_string oc sep; oiter oc sep f t;;
let oiter sep f l = oiter !oc sep f l;;
let rec repeat_string n s acc = 
	if n <= 0 then acc else repeat_string (n-1) s (s ^ acc)
;;	 
let percentage n1 n2 = floor (float_of_int n1 /. float_of_int n2 *. 10000.) /. 100.;;
let print_percentage n1 n2 = os (string_of_float (percentage n1 n2)); os "\\% ";;

(* Output *)
let escape_ s = Str.global_replace (Str.regexp "_") "\\_" s;;

let print_header str ithyl thyl =
	os "\\begin{table}[]"; os "\n"; 
	os "\\centering"; os "\n";
  os ("\\begin{tabular}{@{}" ^ 
	    repeat_string (List.length thyl + 2) "c" "" ^ "@{}}");
	os "\n"; os "\\toprule"; os "\n";
  os ("\\phantom{abc} & " ^ str ^ " & ");
	oiter " & " 
	  (fun x -> os (escape_ x ^ "(" ^ string_of_int (nb_of_ithm ithyl x) ^ ")")) thyl;
  os "\\\\";
	os "\n"; os "\\midrule"; os "\n";
;;

let print_pn_thy lll ulll ithyl pn thy = 
	let n  = nb_of_ithm ithyl thy in
	let np = nb_of_pthm lll pn thy in
	let nu = nb_of_unique ulll pn thy in
	  print_percentage np n; os " "; os (string_of_int nu);
;;
		  
let print_pn lll ulll ithyl thyl pn =
	let ethyl = List.map fst ithyl in
	let n  = sum (List.map (nb_of_ithm ithyl) ethyl) in
	let nu = sum (List.map (nb_of_unique ulll pn) ethyl) in
	let np = sum (List.map (nb_of_pthm lll pn) ethyl) in
	  os (pn ^ " & "); 
		print_percentage np n; os " "; os (string_of_int nu);
		os " & ";
		oiter " & " (print_pn_thy lll ulll ithyl pn) thyl; 
	  os "\\\\"
;;

let print_pnl lll ulll ithyl thyl pnl =
	oiter "\n" (print_pn lll ulll ithyl thyl) pnl; 
	os "\n"; os "\\midrule"; os "\n";
;;

let print_combine_thy cll ithyl thy =
	let n  = nb_of_ithm ithyl thy in
	let n1 = List.assoc thy cll in 
	  print_percentage n1 n
;;

let print_combine_all cll ithyl =
	let ethyl = List.map fst ithyl in
	let n  = sum (List.map (nb_of_ithm ithyl) ethyl) in
	let n1 = sum (List.map (fun x -> List.assoc x cll) ethyl) in
	  print_percentage n1 n

let print_combine cll ithyl thyl =
  os ("total & ");
	print_combine_all cll ithyl; os " & ";
	oiter " & " (print_combine_thy cll ithyl) thyl;
  os "\\\\";
	os "\n"; os "\\bottomrule"; os "\n";
;;

let print_footer str =
	os "\\end{tabular}"; os "\n";
	os ("\\caption{" ^ str ^ "}"); os "\n";
  os "\\label{}"; os "\n";
	os "\\end{table}"; os "\n";
;;
  
let print_table str idir odir pnl thyl =
	let ithyl = mk_ithyl idir in
	let ethyl = List.map fst ithyl in
	let lll = read_stats odir ithyl pnl in
	let compare_pn pn1 pn2 = 
		compare (sum (List.map (nb_of_pthm lll pn2) ethyl)) 
		        (sum (List.map (nb_of_pthm lll pn1) ethyl)) 
	in
	let pnl = List.sort compare_pn pnl in
	let ill = inv_stats lll pnl ithyl in
	let cll = combine_stats ill in
	let ulll = unique_stats_pnl ill pnl in
	  oc := open_out ("latex_stats.txt");
		print_header str ithyl thyl;
		print_pnl lll ulll ithyl thyl pnl; 
		print_combine cll ithyl thyl;
		print_footer str;
		close_out !oc; oc := stdout
;;

(* to compute the first four theory list *)
let rec first_n n l = if l = [] || n <= 0 then [] 
                      else List.hd l :: first_n (n-1) (List.tl l)
;;
let compare_thyl ithyl thy1 thy2=  
	compare (nb_of_ithm ithyl thy2) (nb_of_ithm ithyl thy1)
;;
let testable_thyl idir= 
	let ithyl = mk_ithyl idir in
	  first_n 4 (List.sort (compare_thyl ithyl) (List.map fst ithyl))
;;
	 
(* results
#load "unix.cma";;
#load "str.cma";;
#load "nums.cma";;
#load "latex_stat.cmo";;
open Latex_stat;;
let dir_out = "o";;
let dir_in = "i/f";;
let pnl = all_subdir dir_out;;
let thyl = ["sorting"; "list"; "finite_map"];;
print_table "h4_dep" dir_in dir_out pnl thyl;;
*)
	
			
																																							
