open Hh_parse
open Hh_term
open Toolbox
open Read
open Thf1hh1

let libdir_in = "../../hh2-data/palibs/Coq"
let libdir_out = "../../hh2-data/palibs/Coq-matching"

type coqterm =
  Var of string
| Const of string
| App of coqterm * coqterm
| Lam of coqabstraction
| Case
| Cast
| Fix
| Cofix
| Letin
| Prod of coqabstraction
| IndType of string (* inductive type name *) * coqterm list (* constructors *)
| SortProp | SortSet | SortType
and coqabstraction =  string (* var name *) * coqterm (* var type *) * coqterm (* body *)

let indtype_name indname indnum = 
  indname ^ "_" ^ indnum
let indconstr_name indname constr_num = 
  "__constr_" ^ indname ^ "_" ^ constr_num

let rec to_coqterm tm =
  match tm with
  | Comb(Comb(Id "$Ind", Id "Coq_Init_Logic_True"), Id "0") -> 
    Const("$true")

  | Comb(Comb(Id "$Ind", Id "Coq_Init_Logic_False"), Id "0") ->
    Const("$false")

  | Comb(Comb(Id "$Ind", Id "Coq_Init_Logic_and"), Id "0") ->
    Const("&")

  | Comb(Comb(Id "$Ind", Id "Coq_Init_Logic_or"), Id "0") ->
    Const("|")

  | Comb(Id "$Const", Id "Coq_Init_Logic_not") ->
    Const("~")
  
  | Comb(Id "$Const", Id "Coq_Init_Logic_iff") ->
    Const("=2")

  | Comb(Comb(Id "$Ind", Id "Coq_Init_Logic_eq"), Id "0") ->
    Const("=3")

  | Comb(Comb(Id "$Ind", Id "Coq_Init_Logic_ex"), Id "0") -> 
    Const("?")

  | Comb(Comb(Id "$Ind", Id "Coq_Init_Logic_all"), Id "0") -> 
    Const("!")

  | Comb(Id "$Rel", Id num) -> Var(num)
  | Comb(Id "$Const", Id name) -> Const(name)

  | Comb(Comb(Id "$App", left), args) ->
    let rec build_app left args =
      match args with
      | Comb(args2, arg) ->
        App(build_app left args2, to_coqterm arg)
      | Id "$ConstrArray" ->
        to_coqterm left
      | _ -> failwith "to_coqterm: build_app"
    in
    build_app left args

  | Comb(Comb(Comb(Id "$Lambda", Comb(Id "$Name", Id varname)), vartype), body) ->
    Lam(varname, to_coqterm vartype, to_coqterm body)

  | Comb(Comb(Comb(Comb(Id "$Case", _), _), _), _) -> Case
  | Comb(Comb(Comb(Comb(Id "$LetIn", _), _), _), _) -> Letin
  | Comb(Comb(Comb(Id "$Construct", Id indname), Id indnum), Id constrnum) ->
    Const(indconstr_name (indtype_name indname indnum) constrnum)

  | Comb(Comb(Id "$Cast", trm), ty) -> Cast
  | Comb(Comb(Comb(Id "$Fix", _), _), _) -> Fix
  | Comb(Comb(Id "$Cofix", _), _) -> Cofix
  | Comb(Comb(Comb(Id "$Prod", Comb(Id "$Name", Id varname)), vartype), body) ->
    Prod(varname, to_coqterm vartype, to_coqterm body)
  | Comb(Id "$Sort", Id "$Prop") -> SortProp
  | Comb(Id "$Sort", Id "$Set") -> SortSet
  | Comb(Id "$Sort", Id "$Type") -> SortType

  | Comb(Comb(Id "$Ind", Id indname), Id indnum) ->
    Const(indtype_name indname indnum)
  | _ -> failwith ("to_coqterm: " ^ hh_to_string tm) 

(* Formating *)

(* starts at 1 *)
let rec is_free n tm = match tm with
  | Var(x) -> if (int_of_string x = n) then false else true
  | App(x, y) -> is_free n x && is_free n y
  | Lam(vname, vtype, tm) -> is_free n vtype && is_free (n+1) tm
  | Prod(vname, vtype, tm) -> is_free n vtype && is_free (n+1) tm
  | _ -> true

let rec rm_prod tm = match tm with
  | App(x, y) -> App(rm_prod x, rm_prod y)
  | Lam(vname, vtype, tm) -> Lam(vname, rm_prod vtype, rm_prod tm) 
  | Prod(vname, vtype, tm) -> 
      if is_free 1 tm 
      then Prod("$Anonymous", rm_prod vtype, rm_prod tm)
      else Prod(vname, rm_prod vtype, rm_prod tm)
  | _ -> tm

let variant_name s used =
  if List.mem s used 
  then
    let rec new_name s i =
      let si = s ^ string_of_int i in
      if List.mem si used then new_name s (i + 1)
      else si
    in new_name s 0
  else s

let rec rm_debrujin_aux vl tm = match tm with
  | Var(x) -> Var(List.nth vl (int_of_string x - 1))
  | App(x, y) -> 
      App(rm_debrujin_aux vl x, rm_debrujin_aux vl y)
  | Lam(vname, vtype, tm) ->
     let new_vname = variant_name vname vl in
     Lam(new_vname, rm_debrujin_aux vl vtype, 
         rm_debrujin_aux (new_vname :: vl) tm)
  (* prod anonymous is short for implication should not rename $Anonymous *)
  | Prod("$Anonymous", vtype, tm) -> 
      Prod("$Anonymous", rm_debrujin_aux vl vtype, 
           rm_debrujin_aux ("$Anonymous" :: vl) tm)
  | Prod(vname, vtype, tm) ->
     let new_vname = variant_name vname vl in
     Prod(new_vname, rm_debrujin_aux vl vtype,
          rm_debrujin_aux (new_vname :: vl) tm)
  | _ -> tm

let rm_debrujin tm = rm_debrujin_aux [] tm

let format x = rm_debrujin (rm_prod (to_coqterm x))

(* Printing *)
let write_coqterm out tm =
  let rec write tm =
    match tm with
    | App(Const "~", tm1) -> out "(~ "; write tm1; out ")"
    | App(App(Const op, tm1), tm2) when List.mem op ["&";"|"] -> 
      out_binop op tm1 tm2
    | App(App(Const "=2", tm1), tm2) -> 
      out_binop "=" tm1 tm2
    | App(App(App(Const "=3", tm1), tm2), tm3) -> 
      out_binop "=" tm2 tm3
    | App(Const qt, Lam(vname, vtype, tm)) when List.mem qt ["!";"?"] ->
      out_quant qt vname vtype tm
    | Prod("$Anonymous", vtype, tm) ->
      out_binop "=>" vtype tm
    | Var(x) -> out x
    | Const(c) when List.mem c ["&";"|";"=2";"=3";"!";"?";"~"] -> 
        let new_c = List.assoc c 
          [("&","$and");("|","$or");("=2","$equals2");("=3","$equals3");  
           ("!","$forall");("?","$exists");("~","$not")] 
        in
        out new_c
    | Const(c) -> out c
    | IndType(indname, constr_types) -> out indname
    | App(x, y) -> out "("; write x; out " "; write y; out ")"
    | Lam(vname, vtype, tm) -> out_quant "^" vname vtype tm
    | Case -> out "CASE"
    | Cast -> out "CAST"
    | Fix  -> out "FIX" 
    | Cofix -> out "COFIX"
    | Letin -> out "LETIN"
    | Prod(vname, vtype, tm) -> out_quant "!" vname vtype tm
    | SortType -> out "$t"
    | SortSet ->  out "$t"
    | SortProp -> out "$o"
  and out_quant quant vname vtype tm =
    out ("(" ^ quant ^ "["); out vname; out " : "; write vtype; out "]: "; 
    write tm; out ")"
  and out_binop binop tm1 tm2 =
    out "("; write tm1; out (" " ^ binop ^ " "); write tm2; out ")"
  in
  write tm

let write_coqsort out tm =
  match tm with
  | SortType -> out "ty"
  | SortProp -> out "ax"
  | SortSet  -> out "ty"
  | _        -> write_coqterm print_string tm; 
                failwith "write_human_coqsort"

let coq_to_matching_file (file_in,file_out) =
  mkdir_rec (Filename.dirname file_out);
  let thml1 = Read.read_coq_file file_in in
  let thml2 = List.map (fun (a,b,c,_) -> (format a,to_coqterm b,format c)) thml1 in
  let oc = open_out file_out in 
  let write_thm (a,b,c) =
    output_string oc "tt(";
    write_coqterm (output_string oc) a;
    output_string oc ", ";
    write_coqsort (output_string oc) b; 
    output_string oc ", ";
    write_coqterm (output_string oc) c; 
    output_string oc ").\n"
  in      
  List.iter write_thm thml2;
  close_out oc

let coq_to_matching dir_in dir_out =
  let filel = find_p dir_in in
  let ff file = (dir_in ^ "/" ^ file, dir_out ^ "/" ^ file) in
  let ffl = List.map ff filel in
  List.iter coq_to_matching_file ffl


(* 
rlwrap ./top -I ..

open Coq_hh;;
coq_to_matching libdir_in libdir_out;; 

*)








