(* The HOL Light theorem export can work in two modes: It can write
   OMDoc and it can write THF1. This module implements the common
   functionality. This means that many of the functions are
   parametrized by a "syntax" argument, which can be instantiated
   by one of the modules doing the actual writing. *)

#load "str.cma";;
#load "unix.cma";;


(* HOL Light files always include the complete path, this function
   removes the HOL Light top directory prefix from a file name *)
let noholname s =
  let len = String.length !hol_dir and slen = String.length s in
  if slen > len then
    let sub = String.sub s 0 len in
    if sub = !hol_dir then String.sub s (len + 1) (slen - len - 1)
    else s
  else s
;;

(* This function creates a directory with higher directories, same as "mkdir -p" *)
let rxpslash = Str.regexp "/";;
let mkdir_p fname =
  let path = rev (Str.split rxpslash fname) in
  let mkfolder n sf =
    (try Unix.mkdir sf 0o755 with _ -> ());
    sf ^ "/" ^ n
  in
  ignore (end_itlist mkfolder path)
;;

(* In the OMDoc export each file must refer to its predecessor, so we remember
   the last file that was exported *)
let last_file = ref "";;

(* Recurses over the given hol type and calls the appropriate functions from the given syntax module *)
let rec write_hol_type syn = function
    Tyvar x -> syn.type_var x
  | Tyapp (s, l) -> syn.type_app (write_hol_type syn) s l
;;

(* Finds the instantiation (list of type pairs) needed to instantiate the constant n to type ty *)
let inst_const (n, ty) =
  let gty = get_const_type n in
  let inst = type_match gty ty [] in
  let rinst = map (fun (a, b) -> (b, a)) inst in
  let tvs = tyvars gty in
  map (fun x -> assoc x rinst) tvs
;;

(* Recurses over the given hol (sub-)term and calls the approproate writing functions from the given syntax module syn *)
let rec write_term syn = function
    (Comb (l, r) as t) ->
      let (l,r) = if syn.comb_strip
                  then strip_comb t
                  else (l,[r]) in syn.comb (write_hol_type syn) (write_term syn) l r
  | Var (x, t) -> syn.var (x,t) (fun () -> write_hol_type syn t)
  | Const (s, t) -> syn.const (write_hol_type syn) s (inst_const (s,t))
  | (Abs (l, r) as t) ->
      let (l,r) = if syn.abs_strip then strip_abs t else ([l],r) in
      syn.abs (write_hol_type syn) (write_term syn) (List.map dest_var l) r
;;

(* Write a theorem (or conjecture) and its dependencies using a syntax writer.
   The theorem name must be of format "theory/theorem", e.g. "test/test".
   The role is e.g. Axiom or Definition (see writer_sig.ml). *)
let write_thm (syn : writer_t) (name : string) (tm : term) (role : role_t) (deps : string list) =
  let tvs = map dest_vartype (type_vars_in_term tm)
  and fs = map (fun x -> let v,ty = dest_var x in ((v,ty), fun () -> write_hol_type syn ty)) (frees tm) in
  syn.thm name role deps tvs fs (fun () -> write_term syn tm)
;;

(* Given a syntax, a theorem name association table, a name of the file and a list of history
   entries, writes all the history entries pertaining to this file *)
let do_write syn thn =

  (* Builds an implication out of a list of assumptions and a conclusion *)
  let list_mk_imp (asms, concl) = List.fold_right (fun a sf -> mk_binary "==>" (a, sf)) asms concl in

  (* Writer for a constant introduction *)
  let add_const cname =
    let ty = get_const_type cname in
    let tvs = map dest_vartype (tyvars ty) in
    syn.addconst cname tvs (fun () -> write_hol_type syn ty) in

  (* Writer for a theorem/definition/conjecture statement *)
  let thm name = write_thm syn name (try list_mk_imp (rev_assoc name thn) with _ -> failwith ("writer/thm: " ^ name)) in

  (* Processor for a single history entry *)
  let process_entry = function
      Histhm (name, deps) -> thm name Theorem deps
    | Hisax name -> thm name Axiom []
    | Histy name -> syn.tydef name (get_type_arity name)
    | Hiscon name -> add_const name
    | Hisdef (name, th) -> add_const name; thm th Definition []
    | Histyd (tn, an, rn, t1, t2, deps) ->
        syn.tydef tn (get_type_arity tn);
        add_const an; add_const rn;
        thm t1 Definition deps; thm t2 Definition deps
  in

  (* Final exported function *)
  let process_file fname l =
    let fname = noholname fname in
    if fname <> "tactics.ml" then begin
      syn.thy_start fname (if !last_file = "" then [] else [!last_file]);
      List.iter process_entry l;
      syn.thy_end ();
      last_file := fname
    end
  in

  process_file
;;

