let do_replacements l =
  let l = List.map (fun (p, s) -> (Str.regexp p, s)) l in
  fun s -> List.fold_left (fun sf (r, s) -> Str.global_replace r s sf) s l
;;

let escxml = do_replacements [("&","&amp;"); ("<","&lt;"); (">","&gt;"); ("\"","&quot;"); ("'","&apos;")];;

(* Used for names of Vars and Constants, OMV and OMS *)
let escuri = do_replacements [("%","%25"); ("#","%23"); ("/","%2F"); ("?","%3F")];;

let oc = ref stdout;;
let ocfmt = ref (Format.formatter_of_out_channel !oc);;

let os s = Format.pp_print_string !ocfmt s;;
let ob () = Format.pp_open_box !ocfmt 2;;
let cb () = Format.pp_close_box !ocfmt ();;
let br () = Format.pp_print_cut !ocfmt ();;
let nl () = Format.pp_force_newline !ocfmt ();;

let rec oiter sep f = function
  [] -> () | [e] -> f e | h :: t -> f h; os sep; oiter sep f t;;

let ose s = os (escxml s);;

let tagarg = List.iter (fun (a,b) -> os " "; os a; os "=\""; ose b; os "\"");;
let tag1 n l = br (); os "<"; os n; tagarg l; os "/>";;
let tagob n l = nl (); ob (); os "<"; os n; tagarg l; os ">";;
let tagcb n = cb (); nl (); os "</"; os n; os ">";;
let tago = tagob and tagc = tagcb;;

let do_notation (prec, fi, args) dim2 =
  let fi = ("fixity", fi) and args1 = ("arguments", args) in
  let l = if prec = -1 then [fi; args1] else [fi; args1; "precedence",string_of_int prec] in
  tago "notations" [];
  tag1 "notation" l;
  (if dim2 <> "" then
    let di = ("dimensions", "2") and args2 = ("arguments", dim2) in
    let l = if prec = -1 then [di; fi; args2] else [di; ("fixity", "mixfix"); args2; "precedence",string_of_int prec] in
    tag1 "notation" l else ());
  tagc "notations"
;;

let symbols = [
  "~", "¬";
  "!", "∀";
  "?", "∃";
  "?!", "∃!";
  "@", "ε";
  "lambda", "Λ";
  "minimal", "μ";
  "/\\", "∧";
  "\\/", "∨";
  "==>", "⟹";
  "<=", "≤";
  ">=", "≥";
  "F", "⊥";
  "T", "⊤";
  "NIL", "[]";
  "SUBSET", "⊆";
  "PSUBSET", "⊊";
  "pow", "^";
  "UNION", "∪";
  "INTER", "∩";
  "EMPTY", "∅";
  "IN", "∈";
  "o", "∘"
];;

let notations = [
  "COND", (-1, "mixfix", "if 2 then 3 else 4");
  "UNIV", (-1, "mixfix", "(: 1 )");
  "abs", (-1, "mixfix", "| 1 |");
  "FACT", (-1, "postfix", "0 1 !");
  "fun",  (0, "infix-right", "0 2 →");
  "sum",  (2, "infix-right", "0 2 +");
  "prod", (4, "infix-right", "0 2 ×");
  "cart", (6, "infix-left", "0 2 ^")
];;

let dim2 = [
  "<=_c", "3 ≤ _ c ^ ( %I1 %w %I2 ) 4";
  "<_c",  "3 < _ c ^ ( %I1 %w %I2 ) 4";
  ">=_c", "3 ≥ _ c ^ ( %I1 %w %I2 ) 4";
  ">_c",  "3 > _ c ^ ( %I1 %w %I2 ) 4";
  "=_c",  "3 = _ c ^ ( %I1 %w %I2 ) 4"
];;

let maybe_notation =
  fun name iargs args ->
  let revint = try List.map (fun (a, (b, _)) -> (b, a)) (tl (rev !the_interface)) with _ -> [] in
  let rev_name = try List.assoc name revint with _ -> name in
  let dim2 = try List.assoc rev_name dim2 with _ -> "" in
  try do_notation (List.assoc rev_name notations) dim2 with Not_found ->
    let mkargs a =
      let sargs = string_of_int iargs ^ " " ^ string_of_int a in
      let map_name = try List.assoc rev_name symbols with _ -> rev_name in
      if map_name = name then sargs else sargs ^ " " ^ map_name
    in
    if List.mem rev_name (binders ()) then do_notation (-1, "bindfix-assoc", mkargs 1) dim2 else
    if List.mem rev_name (prefixes ()) then do_notation (-1, "prefix", mkargs 1) dim2 else
    try
      let (p, a) = List.assoc rev_name (infixes ()) in
      let a = if a = "left" then "infix-left" else "infix-right" in
      do_notation (p, a, mkargs 2) dim2
    with Not_found ->
      if iargs = -1 then begin (* it is a type *)
        let args = String.concat " , " (Array.to_list (Array.init args (fun i -> string_of_int (i + 1)))) in
        let astr = if args = "" then "" else "( " ^ args ^ " ) " in
        if astr <> "" then do_notation (-1, "mixfix", astr ^ name) dim2
      end else
        if iargs > 0 or dim2 <> "" or List.mem_assoc rev_name symbols then
          do_notation (-1, "prefix", mkargs 0) dim2 else ()
;;

let url_mmt = "http://cds.omdoc.org/mmt";;
let url_lf = "http://cds.omdoc.org/urtheories";;
let url_hol = "http://latin.omdoc.org/foundations/hollight";;
let url_core = "http://code.google.com/p/hol-light/source/browse/trunk";; (*?r=182*)
let sym base modl name = tag1 "om:OMS" [("base", base); ("module", modl); ("name", name)];;

let lfc name = sym url_lf "LF" name;;
let hlc modl name =
  if modl = "Kernel" or modl = "HOL" then sym url_hol modl name
  else if modl = "Errors" then sym url_mmt modl name
  else sym url_core modl name;;

let tyhash = Hashtbl.create 100;;
let cshash = Hashtbl.create 100;;
let thhash = Hashtbl.create 100;;
let used = Hashtbl.create 100;;

Hashtbl.add tyhash "bool"     ("Kernel", "bool");;
Hashtbl.add tyhash "fun"      ("Kernel", "fun");;
Hashtbl.add tyhash "ind"      ("HOL",    "ind");;
Hashtbl.add cshash "="        ("Kernel", "equal");;
Hashtbl.add cshash "@"        ("HOL",    "@");;
Hashtbl.add cshash "mmterror" ("Errors", "unknown");;

let hl_find_c  s = try let (a, b) = Hashtbl.find cshash s in hlc a b with _ -> failwith ("find_c :" ^ s);;
let hl_find_ty s = try let (a, b) = Hashtbl.find tyhash s in hlc a b with _ -> failwith ("find_ty:" ^ s);;
let hl_find_th s = try let (a, b) = Hashtbl.find thhash s in hlc a b with _ -> failwith ("find_th:" ^ s);;

let current_module = ref "";;

let rec add_to_hash hash s maybes =
  if Hashtbl.mem used maybes then add_to_hash hash s (maybes ^ "_renamed") else
  let escaped = escuri maybes in
  (Hashtbl.add used maybes (); Hashtbl.add hash s (!current_module, escaped); escaped)
;;

let typeo () = tago "type" []; tago "om:OMOBJ" ["xmlns:om","http://www.openmath.org/OpenMath"];;
let typec () = tagc "om:OMOBJ"; tagc "type";;

let omv s = tag1 "om:OMV" ["name", escuri s]

let om_type_var s = omv s;;
let om_type_app oty s = function
    [] -> hl_find_ty s
  | l ->  tago "om:OMA" []; lfc "apply"; hl_find_ty s; List.iter (fun x -> oty x) l; tagc "om:OMA";;

let om_const oty s = function
    [] -> hl_find_c s
  | l -> tago "om:OMA" []; lfc "apply"; hl_find_c s; List.iter oty l; tagc "om:OMA"
;;

let om_comb oty otm h l =
  let llast, lbutlastr = match rev l with h :: t -> (h, t) | _ -> failwith "comb empty" in
  let t1 = list_mk_comb (h, rev lbutlastr) in
  let ty1 = type_of llast and ty2 = snd (dest_fun_ty (type_of t1)) in
  tago "om:OMA" []; lfc "apply"; hlc "Kernel" "Comb"; oty ty1; oty ty2;
  otm t1; otm llast; tagc "om:OMA"
;;

let om_var (s,ty) oty = omv s;;

let om_abs oty otm vs t =
  tago "om:OMA" []; lfc "apply"; hlc "Kernel" "Abs"; oty (snd (hd vs)); oty (type_of t);
  tago "om:OMBIND" []; lfc "lambda"; tago "om:OMBVAR" [];
  tago "om:OMV" ["name",escuri (fst (hd vs))]; tago "type" [];
  tago "om:OMA" []; lfc "apply"; hlc "Kernel" "term"; oty (snd (hd vs)); tagc "om:OMA";
  tagc "type"; tagc "om:OMV"; tagc "om:OMBVAR"; otm t; tagc "om:OMBIND"; tagc "om:OMA"
;;

let om_addconst s tvs oty =
  let renameds = add_to_hash cshash s s in
  tagob "constant" ["name", renameds]; typeo ();
  List.iter (fun tv ->
    tago "om:OMBIND" []; lfc "Pi"; tago "om:OMBVAR" []; tago "om:OMV" ["name",escuri tv];
    tago "type" []; hlc "Kernel" "holtype"; tagc "type";
    tagc "om:OMV"; tagc "om:OMBVAR"
  ) tvs;
  tago "om:OMA" []; lfc "apply"; hlc "Kernel" "term"; oty (); tagc "om:OMA";
  List.iter (fun _ -> tagc "om:OMBIND") tvs;
  typec ();
  maybe_notation s (length tvs) (length (fst (splitlist dest_fun_ty (get_const_type s))));
  tagcb "constant"
;;

let om_tydef s arity =
  let renameds = add_to_hash tyhash s s in
  tagob "constant" ["name", renameds]; typeo ();
  let rec tyd = function
    0 -> hlc "Kernel" "holtype"
  | n -> tago "om:OMA" []; lfc "arrow"; hlc "Kernel" "holtype"; tyd (n - 1); tagc "om:OMA"
  in
  tyd arity;
  typec ();
  maybe_notation s (-1) arity;
  tagcb "constant"
;;


let om_thm s role deps tvs fs otm =
  let fs = zip (map (fun x -> fst (fst x)) fs) (map snd fs) in
  let s = Filename.basename s in
  let renameds = add_to_hash thhash s s in
  tagob "constant" ["name", renameds]; typeo ();
  List.iter (fun tv ->
    tago "om:OMBIND" []; lfc "Pi"; tago "om:OMBVAR" []; tago "om:OMV" ["name",escuri tv];
    tago "type" []; hlc "Kernel" "holtype"; tagc "type";
    tagc "om:OMV"; tagc "om:OMBVAR"
  ) tvs;
  List.iter (fun (s,ty) ->
    tago "om:OMBIND" []; lfc "Pi"; tago "om:OMBVAR" []; tago "om:OMV" ["name",escuri s]; tago "type" [];
    tago "om:OMA" []; lfc "apply"; hlc "Kernel" "term"; ty (); tagc "om:OMA";
    tagc "type"; tagc "om:OMV"; tagc "om:OMBVAR"
  ) fs;
  tago "om:OMA" []; lfc "apply"; hlc "Kernel" "thm"; otm (); tagc "om:OMA";
  List.iter (fun _ -> tagc "om:OMBIND") fs;
  List.iter (fun _ -> tagc "om:OMBIND") tvs;
  typec ();
  if deps <> ["-"] && deps <> ["!"] then begin
    tago "definition" [];
    tago "om:OMA" []; (*lfc "apply";*) hl_find_c "mmterror";
    oiter " " hl_find_th (map Filename.basename deps);
    tagc "om:OMA"; tagc "definition"
  end;
  tagcb "constant"
;;

let om_thy_start oname incl =
  let oname = String.sub oname 0 (String.length oname - 3) in
  print_endline oname;
  current_module := oname;
  let fname = "data/" ^ oname ^ ".omdoc" in
  mkdir_p fname;
  oc := open_out fname;
  ocfmt := Format.formatter_of_out_channel !oc;
  os "<?xml version=\"1.0\" encoding=\"UTF-8\"?>";
  tagob "omdoc" [("xmlns:om","http://www.openmath.org/OpenMath");("xmlns","http://omdoc.org/ns")];
  tagob "theory" [("name",oname);"base", url_core ;("meta",url_hol ^ "?HOL")];
  if incl <> [] then begin
    let oname = hd incl in
    let name = String.sub oname 0 (String.length oname - 3) in
    nl (); tag1 "import" [("from", "?" ^ name)]
  end
;;

let om_thy_end () =
  tagcb "theory"; nl (); tagcb "omdoc";
  Format.pp_print_flush !ocfmt ();
  close_out !oc
;;

let om_syntax = {
  type_var = om_type_var; type_app = om_type_app;
  const = om_const; comb_strip = true; comb = om_comb;
  var = om_var; abs_strip = false; abs = om_abs;
  addconst = om_addconst; tydef = om_tydef;
  thm = om_thm; thy_start = om_thy_start; thy_end = om_thy_end
};;
