(*--------------------------------------------------------------------------
  Escaping
  -------------------------------------------------------------------------- *)

let explode s =
  let rec expl i l =
    if i < 0 then l else
    expl (i - 1) (s.[i] :: l) in
  expl (String.length s - 1) []

let implode l =
  let result = String.create (List.length l) in
  let rec imp i = function
  | [] -> result
  | c :: l -> result.[i] <- c; imp (i + 1) l in
  imp 0 l

let is_tptp_sq_char c = (* does not include dot, colon and vertical bar *)
  let n = Char.code c in
    (40 <= n && n <= 45) || (* dot removed *)
    (48 <= n && n <= 57) || (* colon removed *)
    (59 <= n && n <= 123) || (* vetical bar removed *)
    (125 <= n && n <= 133) ||
    (135 <= n && n <= 176)

let hh_escape s =
  let l1 = explode s in
  let image c =
      if is_tptp_sq_char c 
      then [c]
      else ['|'] @ String.explode (int_to_string (Char.ord c)) @ ['|']
    val l2 = map image l1
  in
    String.implode (List.concat l2)
  end

let squotify name = "'" ^ name ^ "'"

(* Every objects is preceded by its path *)
let remove_ext s = List.hd (Str.split (Str.regexp "\.") s);;

let mk_prefix s = 
  let s1 = noholname s in
  let thy = remove_ext (Filename.basename s1) in
  let l = butlast (Str.split rxpslash s1) in
  String.concat "." (List.map hh_escape (l @ [thy]))

(*--------------------------------------------------------------------------
  Hack to remove hidden theorems
  -------------------------------------------------------------------------- *)

let is_hidden thm_name = 
  try Str.string_before (Filename.basename thm_name) 7 = "hidden_" 
  with _ -> false

(*--------------------------------------------------------------------------
  Dictionnaries
  -------------------------------------------------------------------------- *)

module Sm = Map.Make(struct type t = string let compare = compare end);;
let smfind s m = try Sm.find s m 
  with Not_found -> (print_endline ("Error: smfind " ^ s); failwith "smfind");;

module Stym = Map.Make(struct type t = (string * hol_type) let compare = compare end);;

let ty_names = ref Sm.empty;;
let const_names = ref Sm.empty;;
let var_names = ref Stym.empty;;
let tyvar_names = ref Sm.empty;;
let thm_names = ref Sm.empty;;
let used_names = ref Sm.empty;;

(* No escaping is done *)
let variant_name_map s used =
  try let i = Sm.find s used in
    let rec new_name s i =
      let si = s ^ string_of_int i in
      if Sm.mem si used then new_name s (i + 1)
      else (si, Sm.add s (i + 1) (Sm.add si 0 used))
    in new_name s i
  with Not_found -> (s, Sm.add s 0 used)

let store_name s =
  if Sm.mem s (!used_names) then () 
  else used_names := Sm.add s 0 (!used_names)

(* type *)
let declare_perm_type map s path =
  let s1 = "type." ^ mk_prefix path ^ "." ^ hh_escape s in
  let s2 = squotify s1 in
  store_name s2;
  map := Sm.add s s2 !map;
  s2

(* const *)
let declare_perm_const map s path =
  let s1 = "const." ^ mk_prefix path ^ "." ^ hh_escape s in
  let s2 = squotify s1 in
  store_name s2;
  map := Sm.add s s2 !map;
  s2

(* thm *)
let declare_perm_thm map s =
  let s1 = "thm." ^ mk_prefix (Filename.dirname s) ^ "." ^ 
           hh_escape (Filename.basename s) 
  in
  let s2 = squotify s1 in
  store_name s2; 
  map := Sm.add s s2 !map;
  s2

let declare_fixed map s s2 =
  map := Sm.add s s2 !map; 
  used_names := Sm.add s2 0 !used_names; 
  s2

let _ = ignore (declare_fixed ty_names "bool" "$o")
let _ = ignore (declare_fixed const_names "=" "$equals")

(* nicer names for type variables *)
let prettify_tyvar s = "A"

let declare_temp_tyvar map l =
  let oldmap = !map and oldused = !used_names in
  let fold_fun sl s =
    let (sn, newused) = variant_name_map (prettify_tyvar s) !used_names in
    map := Sm.add s sn !map;
    used_names := newused;
    sn :: sl
  in
  let sl = List.fold_left fold_fun [] l in
  (List.rev sl, fun () -> map := oldmap; used_names := oldused)

let declare_temp_varl map vl =
  let oldmap = !map and oldused = !used_names in
  let fold_fun sl v =
    let (sn, newused) = variant_name_map (prettify_tyvar (fst v)) !used_names in
    map := Stym.add v sn !map;
    used_names := newused;
    sn :: sl
  in
  let sl = List.fold_left fold_fun [] vl in
  (List.rev sl, fun () -> map := oldmap; used_names := oldused)

(*--------------------------------------------------------------------------
  Printing
  -------------------------------------------------------------------------- *)

let oc = ref stdout
let oc_deps = ref stdout
let oc_syn = ref stdout
let os s = output_string !oc s
let rec oiter oc sep f = function
  [] -> () | [e] -> f e | h :: t -> f h; output_string oc sep; oiter oc sep f t
let oiter_deps sep f l = oiter !oc_deps sep f l
let oiter_syn sep f l = oiter !oc_syn sep f l
let oiter sep f l = oiter !oc sep f l

let thf_type_var s = os (smfind s !tyvar_names)
let thf_type_app oty s = function
  [] -> os (smfind s !ty_names)
| l -> if s = "fun" then begin
  os "("; oty (hd l); os " > "; oty (hd (tl l)); os ")"
end else begin
  os "("; os (smfind s !ty_names); List.iter (fun x -> os " @ "; oty x) l; os ")"
end;;

let rec thf_const oty s = function
  [] -> os (smfind s !const_names)
| l -> os "("; os (smfind s !const_names); List.iter (fun x -> os " @ "; oty x) l; os ")";;


let thf_comb oty otm h l =
  if is_const h && fst (dest_const h) = "=" && List.length l = 2 then begin
    os "("; otm (hd l); os " = "; otm (hd (tl l)); os ")"
  end else if is_const h && fst (dest_const h) = "/\\" && List.length l = 2 then begin
    os "("; otm (hd l); os " & "; otm (hd (tl l)); os ")"
  end else if is_const h && fst (dest_const h) = "\\/" && List.length l = 2 then begin
    os "("; otm (hd l); os " | "; otm (hd (tl l)); os ")"
  end else if is_const h && fst (dest_const h) = "==>" && List.length l = 2 then begin
    os "("; otm (hd l); os " => "; otm (hd (tl l)); os ")"
  end else if is_const h && fst (dest_const h) = "~" && List.length l = 1 then begin
    os "(~("; otm (hd l); os "))"
  end else if is_const h && fst (dest_const h) = "!" && List.length l = 1 && is_abs (hd l) then begin
    let vl, t = strip_forall (list_mk_comb (h, l)) in 
    let vl = List.map dest_var vl in
    let tl = List.map snd vl in
    let nl, undeclare = declare_temp_varl var_names vl in
      os ("(!["); oiter ", " (fun (n, ty) -> os n; os " : "; oty ty) (zip nl tl); 
      os "]: "; otm t; os ")";
      undeclare ()
  end else if is_const h && fst (dest_const h) = "?" && List.length l = 1 && is_abs (hd l) then begin
    let vl, t = strip_exists (list_mk_comb (h, l)) in
    let vl = List.map dest_var vl in
    let tl = List.map snd vl in
    let nl, undeclare = declare_temp_varl var_names vl in
      os ("(?["); oiter ", " (fun (n, ty) -> os n; os " : "; oty ty) (zip nl tl); 
      os "]: "; otm t; os ")";
      undeclare ()
  end else begin
    os "("; otm h; List.iter (fun x -> os " @ "; otm x) l; os ")"
  end

let thf_var (s,ty) oty = os (Stym.find (s,ty) !var_names);;

let thf_abs oty otm vl t =
  let tl = List.map snd vl in
  let nl, undeclare = declare_temp_varl var_names vl in
  os "(^["; oiter ", " (fun (n, ty) -> os n; os " : "; oty ty) (zip nl tl); os "]: "; otm t; os ")";
  undeclare ()
;;

let rec rev_interface n = function
    [] -> n
  | (a, (b, _)) :: t -> if b = n then a else rev_interface n t;;

let thf_addconst origs tvs oty =
  let news = match origs with
  | "!"   -> declare_fixed const_names origs "$forall"
  | "?"   -> declare_fixed const_names origs "$exists"
  | "/\\" -> declare_fixed const_names origs "$and"
  | "\\/" -> declare_fixed const_names origs "$or"
  | "==>" -> declare_fixed const_names origs "$imply"
  | "~"   -> declare_fixed const_names origs "$not"
  | "T"   -> declare_fixed const_names origs "$true"
  | "F"   -> declare_fixed const_names origs "$false"
  | _     -> declare_perm_const const_names origs 
             (assoc origs (!the_term_paths))
  in
  let tvs, undeclare = declare_temp_tyvar tyvar_names tvs in
  let newsp = if news.[0]='\'' then String.sub news 1 (String.length news - 1) else news ^ "'" in
  os "thf('thf_const_"; os newsp; os ", type, "; os news; os " : ";
  begin match tvs with
    [] -> ()
  | l -> os "!>["; oiter ", " (fun x -> os x; os " : $tType") l; os "]: "
  end;
  oty (); os ").\n"; undeclare ();
  let origs = rev_interface origs !the_interface in
  let notation =
    if List.mem origs (binders ()) then ["binder"] else
    if List.mem origs (prefixes ()) then ["prefix"] else
    try let (p, a) = List.assoc origs (infixes ()) in
        let n = if a = "right" then "infixr" else "infixl" in [n; string_of_int p]
    with Not_found -> []
  in
  if notation <> [] then (oiter_syn " " (output_string !oc_syn) (origs :: notation); output_string !oc_syn "\n")
;;

let thf_tydef s arity =
  let s = declare_perm_type ty_names s (assoc s (!the_type_paths)) in
  let sp = if s.[0]='\'' then String.sub s 1 (String.length s - 1) else s ^ "'" in
  os "thf('thf_type_"; os sp; os ", type, "; os s; os " : ";
  let rec tyd = function
      0 -> os "$tType"
    | n -> os "$tType > "; tyd (n - 1)
  in
  tyd arity; os ").\n"
;;

(* TODO @Cezary: this models the previous behaviour, check if that is really desired! *)
let string_of_role = function
  | Axiom -> "definition"
  | Definition -> "definition"
  | Conjecture -> "conjecture"
  | Theorem -> "axiom"
;;

let thf_thm name role deps tvs fs otm =
  if is_hidden name then () (* do not write definitions of hidden theorems *)
  else 
 begin
  let name = declare_perm_thm thm_names name in
  let tvn, undeclare = declare_temp_tyvar tyvar_names tvs in
  let vl, vt = List.split fs in
  let vn, undeclare2 = declare_temp_varl var_names vl in

  os "thf("; os name; os ", "; os (string_of_role role); os ", ";

  if List.length tvs > 0 or List.length vn > 0 then begin
    os "(![";
    let tvn = List.map (fun s -> (s, fun () -> os "$tType")) tvn in
    let vn = zip vn vt in
    oiter ", " (fun (s, t) -> os s; os " : "; t ()) (tvn @ vn); os "]: "
  end;
  otm ();
  if List.length vn > 0 then undeclare2 ();
  if List.length tvn > 0 then undeclare ();
  if List.length vn > 0 or List.length tvs > 0 then os ")";
  os ").\n";
  (* Write dependencies only for provable theorems: no hidden ones *)
  if not (List.exists is_hidden deps) then begin
    let deps2 = List.map (fun x -> smfind x !thm_names) deps in
    oiter_deps " " (output_string !oc_deps) (name :: deps2);
    output_string !oc_deps "\n"
  end
 end
;;

let thf_thy_start fname _ =
  print_endline fname;
  let fname = "data/" ^ (String.sub fname 0 (String.length fname - 2)) in
  mkdir_p fname;
  oc := open_out (fname ^ "p");
  oc_deps := open_out (fname ^ "hd");
  oc_syn := open_out (fname ^ "sx")
;;

let thf_thy_end () =
  close_out !oc; oc := stdout;
  close_out !oc_deps; oc_deps := stdout;
  close_out !oc_syn; oc_syn := stdout;;

let thf_syntax = {
  type_var = thf_type_var; 
  type_app = thf_type_app;
  const = thf_const; 
  comb_strip = true; 
  comb = thf_comb;
  var = thf_var; 
  abs_strip = true; 
  abs = thf_abs;
  addconst = thf_addconst; 
  tydef = thf_tydef;
  thm = thf_thm;
  thy_start = thf_thy_start; 
  thy_end = thf_thy_end
};;


