type coq_parse = 
  | Var    of string
  | Const  of string * string
  | Apply  of coq_parse list
  | Lambda of ((string * coq_parse) list * coq_parse) 
  | Prod   of ((string * coq_parse) list * coq_parse) 

(* Writing tools *)
let mkdir s = try Unix.mkdir s 0o777 with Unix.Unix_error (Unix.EEXIST,_,_) -> ()
let rec mk_dir s = 
  if s = "." || s = ".." || s = "" || s = "/" then ()
  else (mk_dir (Filename.dirname s); mkdir s)

let append_string file s =
  mk_dir (Filename.dirname file);  
  let oc = open_out_gen [Open_creat; Open_text; Open_append] 0o644 file in 
  Printf.fprintf oc "%s\n" s; 
  close_out oc

let append_stringl file sl =
  mk_dir (Filename.dirname file);  
  let oc = open_out_gen [Open_creat; Open_text; Open_append] 0o644 file in 
  List.iter (Printf.fprintf oc "%s\n") sl; 
  close_out oc

let clean_dir dir =
  mkdir dir; ignore (Sys.command ("rm -r " ^ dir)); mkdir dir

let cic_of file = ("cic:/matita/" ^ file)
let rm_cic file = try 
                    if Str.first_chars file 12 = "cic:/matita/" 
                    then Str.string_after file 12
                    else failwith ("rm_cic: " ^ file) 
                  with _ -> failwith ("rm_cic: " ^ file) 

(* Escaping *)
let escape_prime s =  Str.global_replace (Str.regexp "'") "\\'" s

let escape_slash s =
  let s1 = Str.global_replace (Str.regexp "#") "#hash#" s in
  let s2 = Str.global_replace (Str.regexp "/") "#slash#" s1 in
  let s3 = Str.global_replace (Str.regexp "\"") "#quote#" s2 in
  let s4 = Str.global_replace (Str.regexp "\\") "#bslash#" s3 in
  s4


let is_lowercase c = 
  let i = Char.code c in i >= Char.code 'a' && i <= Char.code 'z' 
let is_uppercase c = 
  let i = Char.code c in i >= Char.code 'A' && i <= Char.code 'Z'
let is_numeric c =
  let i = Char.code c in i >= Char.code '0' && i <= Char.code '9'
let is_alphanumeric c =
  (is_lowercase c) or (is_uppercase c) or (is_numeric c) or 
  (Char.code c == Char.code '_')


let can f s = try (f s; true) with _ -> false

let is_tptp s =
  let f c = if is_alphanumeric c then () else raise Exit in
  can (String.iter f) s || List.mem s ["$t";"$o";"=>"]

let tptp_escape s =
  if is_tptp s then s else "'" ^ escape_prime (escape_slash s) ^ "'"


(* Transform into Coq terms *)
let tag_of tm = match tm with 
  | Xml.Element (s,_,_) -> s
  | _ -> failwith "tag_of" 

let idl_of tm = match tm with 
  | Xml.Element (_,idl,_) -> idl
  | _ -> failwith "idl_of" 

let dest_target tm = match tm with
  | Xml.Element ("target",_,[t]) -> t 
  | _ -> failwith "dest_target"

let last x = List.hd (List.rev x)
let butlast x = List.rev (List.tl (List.rev x)) 


(* Inductive definition *)
(* TODO use proper constructor name *)

let dest_ctor tm = match tm with
  | Xml.Element ("Constructor",idl,[t]) -> t
  | _ -> failwith ("dest_ctor: " ^ Xml.to_string_fmt tm)

let rec dest_ctorl n path tml = match tml with
  | [] -> []
  | t :: m -> ((path,string_of_int n),dest_ctor t) :: dest_ctorl (n+1) path m

let dest_arity tm = match tm with
  | Xml.Element ("arity",_,[t]) -> t
  | _ -> failwith ("dest_arity: " ^ Xml.to_string_fmt tm)
 
let dest_indtype path tm = match tm with
  | Xml.Element ("InductiveType",idl,arity :: cl) ->
     (path, dest_arity arity, dest_ctorl 1 path cl)
  | _ -> failwith ("dest_indtype: " ^ Xml.to_string_fmt tm)

let dest_indl path tm = match tm with
  | Xml.Element ("InductiveDefinition",_, at :: tl) -> List.map (dest_indtype path) tl
  | _ -> failwith ("dest_inddefl: " ^ Xml.to_string_fmt tm)


(* Constant *)
let dest_con file tm = match tm with
  | Xml.Element ("ConstantType",_,
      [Xml.Element("attributes",_,
        [Xml.Element("flavour",flavourl,_)]
      );t]) 
    ->
      (file,   
      begin
      match List.assoc "value" flavourl with 
      | "theorem" | "variant" | "lemma" | "axiom" -> "ax"
      | "definition" | "mutual_definition" -> "ty"
      | x            -> failwith ("Unknown flavour: " ^ Xml.to_string_fmt tm)
      end
      , t)
  | Xml.Element ("ConstantType",idl,[Xml.Element("attributes",_,_);t]) 
      -> (file,"ax",t) 
  | _ -> failwith ("dest_top: " ^ Xml.to_string_fmt tm)

(* Constant Body *)
let dest_con_body tm = match tm with
  | Xml.Element ("ConstantBody",idl,[t]) -> (List.assoc "for" idl,t)
  | _ -> failwith ("dest_con_body: " ^ Xml.to_string_fmt tm)

(* WIP 
let dest_def tm = match tm with 
  | Xml.Element ("def",idl,[t]) -> (List.assoc "binder" idl, t)
  | _ -> failwith "dest_def"
*)

(* WIP
let (v,t') = dest_def def in
      Apply [Const ("LETIN.logical",""); 
Lambda ([("A",Const ("Type.sort","")]Lambda(v,xml_to_coq t',xml_to_coq t))] 
*)
(* WIP
let argument tm = match tm with
  | Xml.Element (_,_,[t]) -> t
  | _ -> failwith ("argument: " ^ Xml.to_string_fmt tm)

let dest_fix tm = match tm with
  | Xml.Element ("FIX",_,[Xml.Element("FixFunction",idl,[ty;body])] ) -> 
     (List.assoc "name" idl, argument ty, argument body)
  | _ -> failwith ("dest_fix: " ^ Xml.to_string_fmt tm)  

let (name,ty,body) = dest_fix tm in
      Apply [Const ("FIX.logical",name); xml_to_coq ty; xml_to_coq body]
*)


let rec xml_to_coq tm = match tm with
  | Xml.Element ("LETIN",_,_) -> Const ("LETIN.logical","")
  | Xml.Element ("CAST",_,_) -> Const ("CAST.logical","")
  | Xml.Element ("FIX",_,_) -> Const ("FIX.logical","")
  | Xml.Element ("COFIX",_,_) -> Const ("COFIX.logical","")
  | Xml.Element ("PROD",_,tml) -> 
      Prod (List.map decl_to_coq (butlast tml), 
            xml_to_coq (dest_target (last tml)))
  | Xml.Element ("LAMBDA",_,tml) -> 
      Lambda (List.map decl_to_coq (butlast tml), 
              xml_to_coq (dest_target (last tml)))
  | Xml.Element ("APPLY",_,tml)  -> Apply (List.map xml_to_coq tml)
  | Xml.Element ("CONST",_,_) 
  | Xml.Element ("MUTIND",_,_) -> Const (rm_cic (List.assoc "uri" (idl_of tm)), "") 
  | Xml.Element ("MUTCASE",_,_) -> Const ("CASE.logical","")
(* Const (rm_cic (List.assoc "uriType" (idl_of tm)), "Case") *)
  | Xml.Element ("MUTCONSTRUCT",_,_) -> 
      Const (rm_cic (List.assoc "uri" (idl_of tm)), List.assoc "noConstr" (idl_of tm))
  | Xml.Element ("SORT",_,_) -> 
      let s = (List.assoc "value" (idl_of tm)) in
        if (try Str.first_chars s 4 = "Type" with _ -> false) 
        then Const ("Type.sort","")
        else Const ( s ^ ".sort","")
  | Xml.Element ("REL",idl,_) -> Var (List.assoc "binder" idl)
  | _ -> failwith ("xml_to_coq:" ^ Xml.to_string_fmt tm)
and decl_to_coq tm = match tm with
  | Xml.Element ("decl",idl, [t]) 
    when List.mem "binder" (List.map fst idl) -> 
      (List.assoc "binder" idl, xml_to_coq t)
  | Xml.Element ("decl",idl, [t]) -> 
      (List.assoc "id" idl, xml_to_coq t)
  | _ -> failwith ("decl_to_coq:" ^ Xml.to_string_fmt tm)


(* Modify Coq terms *)
let rec sub_tm change tm = match tm with
  | Var s         -> Var s
  | Const (s1,s2) -> Const (s1,s2)
  | Apply l       -> Apply (List.map change l)
  | Lambda (l,t)  -> Lambda (sub_tm_decl change l,change t)
  | Prod  (l,t)   -> Prod (sub_tm_decl change l,change t)
and sub_tm_decl change l =
  let f (s,t) = (s,change t) in List.map f l

let rec var_list tm = match tm with
  | Var s -> [s]
  | Const _ -> []
  | Apply l      -> List.concat (List.map var_list l)
  | Lambda (l,t) -> var_list_decl l @ var_list t
  | Prod  (l,t)  -> var_list_decl l @ var_list t 
and var_list_decl l =
  let f (s,t) = var_list t in List.concat (List.map f l)

let rec expand_lambda tm = match tm with
  | Lambda ([],_)          -> failwith "expand_lambda"
  | Lambda ([(s,t')],t)    -> Lambda ([(s,expand_lambda t')], expand_lambda t)
  | Lambda ((s,t') :: m,t) -> Lambda ([(s,expand_lambda t')], 
                                      expand_lambda (Lambda (m,t))) 
  | _                     -> sub_tm expand_lambda tm

let rec expand_prod tm = match tm with
  | Prod ([],_)          -> failwith "expand_prod"
  | Prod ([(s,t')],t)    -> Prod ([(s,expand_prod t')], expand_prod t)
  | Prod ((s,t') :: m,t) -> Prod ([(s,expand_prod t')], expand_prod (Prod (m,t))) 
  | _                    -> sub_tm expand_prod tm

let rec prod_to_imp tm = match tm with
  | Prod ([(s,t')],t) -> if List.mem s (var_list t) 
                         then sub_tm prod_to_imp tm
                         else Apply ([Const ("=>.logical",""); prod_to_imp t'; prod_to_imp t])
  | _                 -> sub_tm prod_to_imp tm 

let rec expand_apply tm = match tm with
  | Apply l  -> expand_apply_list (Apply (List.rev l))
  | _        -> sub_tm expand_apply tm
and expand_apply_list tm = match tm with 
  | Apply [t]      -> expand_apply t
  | Apply (t :: m) -> Apply [expand_apply_list (Apply m); expand_apply t]
  | _ -> failwith "expand_apply_list"

(* Create dictionaries for readable constants and variables *)
let used_names = ref []
let const_names = Hashtbl.create 20000
let var_names = Hashtbl.create 200

let variant_name s used =
  try
    let i = List.assoc s !used in 
    let rec name s i =
      let si = s ^ string_of_int i in
        if List.mem si (List.map fst !used) then name s (i + 1)
        else (used := (s,(i + 1)):: (si,0) :: !used; si)
    in 
      name s i 
  with Not_found -> (used := (s,0) :: !used; s)

let declare_perm ((s1,s2) as ss) =
  try tptp_escape (Hashtbl.find const_names ss) with Not_found ->
    let s3 = (Filename.chop_extension (Filename.basename s1) ^ s2) in
    let s4 = match s3 with
               "Type" -> "$t"
             | "Prop" -> "$o" 
             | "Set"  -> "$t" 
             | "=>"   -> "=>"
             | _      -> variant_name s3 used_names
    in
    Hashtbl.add const_names ss s4; 
    tptp_escape s4

let declare_var s =
  let s' = variant_name s used_names in
  Hashtbl.add var_names s s'

let declare_perml sl = List.iter (fun x -> ignore (declare_perm x)) sl
let declare_varl sl = List.iter declare_var sl
 
module StrSet = Set.Make(String)
module Ss = struct
  type t = string * string
  let compare a b = compare a b
end
module SsSet = Set.Make(Ss)

let rec get_cset tm = match tm with
  | Var s               -> SsSet.empty 
  | Const (s1,s2)       -> SsSet.add (s1,s2) SsSet.empty
  | Lambda ([(s,t')],t) -> SsSet.union (get_cset t') (get_cset t)
  | Prod ([(s,t')],t)   -> SsSet.union (get_cset t') (get_cset t)
  | Apply [t1;t2]       -> SsSet.union (get_cset t1) (get_cset t2)
  | _                   -> failwith "get_cset"

let rec get_vset tm = match tm with
  | Var s               -> StrSet.add s StrSet.empty
  | Const _             -> StrSet.empty
  | Lambda ([(s,t')],t) -> StrSet.add s (StrSet.union (get_vset t') (get_vset t))
  | Prod ([(s,t')],t)   -> StrSet.add s (StrSet.union (get_vset t') (get_vset t))
  | Apply [t1;t2]       -> StrSet.union (get_vset t1) (get_vset t2)
  | _                   -> failwith "get_vset"

let find_var s = try tptp_escape (Hashtbl.find var_names s) 
  with Not_found -> failwith ("find_var: " ^ s)
let find_const (s1,s2) = try tptp_escape (Hashtbl.find const_names (s1,s2)) 
  with Not_found -> failwith ("find_const: " ^ s1 ^ " " ^ s2)

let rec rename_aux tm = match tm with
  | Var s               -> Var (find_var s)
  | Const (s1,s2)       -> Const (find_const (s1,s2),"")
  | Lambda ([(s,t')],t) -> 
    Lambda ([(find_var s, rename_aux t')],rename_aux t)
  | Prod ([(s,t')],t)   -> 
    Prod ([(find_var s, rename_aux t')], rename_aux t)
  | Apply [t1;t2]       -> Apply ([rename_aux t1; rename_aux t2])
  | _                   -> failwith "rename_aux"

let rename tm =
  declare_perml (SsSet.elements (get_cset tm));
  let used_mem = !used_names in
  Hashtbl.clear var_names; 
  declare_varl (StrSet.elements (get_vset tm));
  let t = rename_aux tm in
  used_names := used_mem; t   

(* Dependencies *)

(* Translate to HH terms *)
let rec coq_to_hh tm = match tm with
  | Var s               -> Hh_parse.Id s
  | Const (s1,s2)       -> Hh_parse.Id s1
  | Apply [t1;t2]       -> Hh_parse.Comb (coq_to_hh t1,coq_to_hh t2)
  | Lambda ([(s,t')],t) -> Hh_parse.Abs (s,coq_to_hh t',coq_to_hh t)
  | Prod ([(s,t')],t)   -> Hh_parse.Comb 
      (Hh_parse.Id "!", Hh_parse.Abs (s, coq_to_hh t', coq_to_hh t)) 

  | _                   -> failwith "coq_to_hh"

(* Print *)
let rec strip_quant q (bvl,tm) = match tm with
  | Hh_parse.Comb (Hh_parse.Id q, Hh_parse.Abs (n,ty,t)) -> 
      strip_quant q (bvl @ [(n,ty)], t)
  | _ -> (bvl,tm)

let rec strip_abs (bvl,tm) = match tm with
  | Hh_parse.Abs (n,ty,t) -> strip_abs (bvl @ [(n,ty)], t)
  | _ -> (bvl,tm)

let infix_symb = [("And","&");("Or","|");("=>","=>")]

let rec print_hh tm = match tm with
  | Hh_parse.Comb (Hh_parse.Comb(Hh_parse.Id op,x),y) 
    when List.mem op (List.map fst infix_symb) -> 
      "(" ^ print_hh x ^ " " ^ List.assoc op infix_symb ^ " " ^ print_hh y ^ ")"
  | Hh_parse.Comb (Hh_parse.Comb (Hh_parse.Comb (Hh_parse.Id "eq",_),x),y) -> 
      "(" ^ print_hh x ^ " = " ^ print_hh y ^ ")"
  | Hh_parse.Comb(Hh_parse.Id "Not",x) -> "(~ " ^ print_hh x ^ ")" 
  | Hh_parse.Comb (Hh_parse.Id "!", Hh_parse.Abs _)  ->
      let (bvl,t) = strip_quant "!" ([],tm) in
        "(![" ^ print_bvl bvl ^ "]: " ^ print_hh t ^ ")"
  | Hh_parse.Comb (x,y) -> "(" ^ print_hh_comb tm ^ ")"
  | Hh_parse.Abs _ -> let (bvl,t) = strip_abs ([],tm) in
     "(^[" ^ print_bvl bvl ^ "]: " ^ print_hh t ^ ")"
  | Hh_parse.Id x -> x
and print_hh_comb tm = match tm with
  | Hh_parse.Comb(x,y) -> print_hh_comb x ^ " " ^ print_hh y
  | _                  -> print_hh tm
and print_bvl bvl = match bvl with
  | [] -> failwith "print_bvl"
  | [(n,ty)] -> n ^ ": " ^ print_hh ty
  | (n,ty) :: m ->  n ^ ": " ^ print_hh ty ^ ", " ^ print_bvl m

(* SUMMARY *)
let parse_xml file =
  let sl = Read.file_to_stringl file in
  let r1 = Str.regexp "[<][?]xml.*[?][>]" in
  let r2 = Str.regexp "[<][!]DOCTYPE.*[>]" in
  let f s = Str.global_replace r2 "" (Str.global_replace r1 "" s) in
  let sl1 = List.map f sl in
    Xml.parse_string (String.concat "" sl1)

let modify_coq t =
  let t1 = expand_prod (expand_lambda t) in
  let t2 = prod_to_imp t1 in
  let t3 = expand_apply t2 in
   rename t3

let xml_to_hh t = coq_to_hh (modify_coq (xml_to_coq t))

(* ORDER on CONSTANTS *)
let uniq l =
  let h = Hashtbl.create (List.length l) in
  List.iter (fun x -> Hashtbl.replace h x ()) l;
  Hashtbl.fold (fun x () xs -> x :: xs) h []

let rec all_uri_aux tm = match tm with
  | Xml.Element (_,idl,tml) -> 
      let l = (try [List.assoc "uri" idl] with Not_found -> []) @
              (try [List.assoc "uriType" idl] with Not_found -> [])

      in
      l @ List.concat (List.map all_uri_aux tml)
  | _ -> failwith ("all_uri_aux: " ^ Xml.to_string_fmt tm)  

let all_uri tm = uniq (all_uri_aux tm)

let const_dep = Hashtbl.create 2000
let const_td = Hashtbl.create 2000
let thy_dep = Hashtbl.create 500
let thy_td = Hashtbl.create 500

let update_dep dir_in file =
  let dep = List.filter (fun x -> x <> file) 
    (List.map rm_cic (all_uri (parse_xml (dir_in ^ "/" ^ file ^ ".xml")))) in
  let thy = Filename.dirname file in
  let thyl = List.filter (fun x -> x <> thy) (List.map Filename.dirname dep) in 
  let pthyl = try Hashtbl.find thy_dep thy with Not_found -> [] in
  Hashtbl.add const_dep file dep;
  Hashtbl.replace thy_dep thy (uniq (pthyl @ thyl))

let init_dep dir_in filel = 
  Hashtbl.clear const_dep;
  Hashtbl.clear const_td;
  Hashtbl.clear thy_dep;
  Hashtbl.clear thy_td;
  List.iter (update_dep dir_in) filel

(* Constant sorting *)
let rec td_of_check cog c =   
  if c = cog then failwith ("Constant loop: " ^ c) else
  try Hashtbl.find const_td c with Not_found ->  
  begin 
    let cl = try Hashtbl.find const_dep c 
             with Not_found -> failwith ("Undeclared constant: " ^ c) in
    let res =  uniq (List.concat (cl :: List.map (td_of_check cog) cl)) in
    Hashtbl.add const_td c res; res
  end

let rec td_of c =   
  try Hashtbl.find const_td c with Not_found ->  
  begin 
    let cl = try Hashtbl.find const_dep c 
             with Not_found -> failwith ("Undeclared constant: " ^ c) in
    let res =  uniq (List.concat (cl :: List.map (td_of_check c) cl)) in
    Hashtbl.add const_td c res; res
  end

let rec topo_sort cl = match cl with 
  | [] -> []
  | c :: m -> let l = td_of c in
              let (l1,l2) = List.partition (fun x -> List.mem x l) m in 
              topo_sort l1 @ [c] @ topo_sort l2

(* Theory sorting *)
let rec td_thy_of_check thyog thy =   
  if thy = thyog then failwith ("Theory loop: " ^ thy) else
  try Hashtbl.find thy_td thy with Not_found ->  
  begin 
    let thyl = Hashtbl.find thy_dep thy in
    let res =  uniq (List.concat (thyl :: List.map (td_thy_of_check thyog) thyl)) in
    Hashtbl.add thy_td thy res; res
  end

let rec td_thy_of thy =   
  try Hashtbl.find thy_td thy with Not_found ->  
  begin 
    let thyl = Hashtbl.find thy_dep thy in
    let res =  uniq (List.concat (thyl :: List.map (td_thy_of_check thy) thyl)) in
    Hashtbl.add thy_td thy res; res
  end

let rec topo_sort_thy thyl = match thyl with 
  | [] -> []
  | thy :: m -> let l = td_thy_of thy in
                let (l1,l2) = List.partition (fun x -> List.mem x l) m in 
                topo_sort_thy l1 @ [thy] @ topo_sort_thy l2


let write_thygraph dir_out file =
  let f thy thyl = append_string (dir_out ^ "/" ^ file) (String.concat " " (thy :: thyl)) in
  Hashtbl.iter f thy_dep

let alist_of_hash hash = 
    Hashtbl.fold (fun k v l -> (k,v) :: l) hash  [] 

let write_thyorder dir_out file =
  let thyl = List.map fst (alist_of_hash thy_dep) in
  let rthyl = topo_sort_thy thyl in
    append_string (dir_out ^ "/" ^ file) (String.concat " " rthyl)

let write_constnames dir_out file =
  let f (k1,k2) v = append_string (dir_out ^ "/" ^ file) 
                      (k1 ^ " " ^ k2 ^ " " ^ v) 
  in
  Hashtbl.iter f const_names

let tt_string (path,role,tm) =
  "tt(" ^ declare_perm (path,"") ^ "," ^ role ^ "," ^ print_hh (xml_to_hh tm) ^ ")."

let tt_ax (path,tm) = tt_string (path,"ax",tm)
let tt_ty (path,tm) = tt_string (path,"ty",tm)
let tt_ctor ((path,n),tm) = 
  "tt(" ^ declare_perm (path,n) ^ "," ^ "ty" ^ "," ^ print_hh (xml_to_hh tm) ^ ")."

let con_body_to_hhs file = 
  let (cpath,tm) = dest_con_body (parse_xml file) in
  tt_ty (cpath,tm)
 
let con_to_hhs dir_in file = 
  tt_string (dest_con file (parse_xml (dir_in ^ "/" ^ file ^ ".xml")))

let ind_to_hhsl dir_in file  =
  let l = dest_indl file (parse_xml (dir_in ^ "/" ^ file ^ ".xml")) in
  let f (path,t,ctorl) = tt_ty (path,t) :: List.map tt_ctor ctorl in
    List.concat (List.map f l)

let outfile dir_out ext file =
  if Filename.dirname file = "." 
  then dir_out ^ "/topdirectory" ^ ext
  else dir_out ^ "/" ^ Filename.dirname file ^ ext
    
let write_con_ind dir_in dir_out (file,con_ind) = 
  let file_out = outfile dir_out ".p" file in
  match con_ind with
  | "con" -> append_string file_out (con_to_hhs dir_in file)
  | "ind" -> append_stringl file_out (ind_to_hhsl dir_in file)
  | _ -> failwith "write_con_ind"

let coqdir_to_hhdir dir_in dir_out = 
  clean_dir dir_out;
  Hashtbl.clear const_names; used_names := [];
  let l1 = List.map Filename.chop_extension (Read.find_con_xml dir_in) in
  let l2 = List.map Filename.chop_extension (Read.find_ind_xml dir_in) in
  let l = l1 @ l2 in
  init_dep dir_in l;
  let rl = topo_sort l in
  let rl' = List.map (fun x -> if List.mem x l1 then (x,"con") else (x,"ind")) rl in
  List.iter (write_con_ind dir_in dir_out) rl';
  write_thygraph dir_out "thygraph";
  write_thyorder dir_out "thyorder";
  write_constnames dir_out "constnames"


(* WIP proof terms    
  let l3 = pair_filel dir_in dir_out ".pt" (Read.find_con_body_xml dir_in) in
List.iter write_con_body l3;
*)




