(*-------------------------------------------------------------------------- *)
(*-------------------------------------------------------------------------- *)

(*-------------------------------------------------------------------------- 
  Reading a file
  -------------------------------------------------------------------------- *)

let file_iter fname fn =
  let ic = try open_in fname with Sys_error _ -> failwith ("file_iter: "^fname) in
  let next = ref 0 in
  let rec suck_lines () = fn !next (input_line ic); incr next; suck_lines () in
  try suck_lines () with End_of_file -> close_in ic

let file_to_stringl file =
  let r = ref [] in
  let f _ s = r := s :: !r in
    file_iter file f; List.rev !r

(*-------------------------------------------------------------------------- 
  Recursive search of files with a given extension
  -------------------------------------------------------------------------- *)

let rec rdirents isok prefix acc d =
  try
    let dirh = Unix.opendir (prefix ^ d) in
    let rec fs acc = try fs
      (
      let l = Unix.readdir dirh in 
      if l <> "." && l <> ".." 
        then rdirents isok (prefix ^ d ^ "/") acc l 
        else acc
      )
      with End_of_file -> acc in
    let ret = fs acc in
    Unix.closedir dirh; ret
  with Unix.Unix_error (Unix.ENOTDIR, _, _) -> if isok d then (prefix ^ d) :: acc else acc

let find_ext dir ext =
  let n_ext = String.length ext in
  let isok s = 
    let l = String.length s in 
    l > n_ext && String.sub s (l - n_ext) n_ext = ext 
  in
  let l = rdirents isok "" [] dir in
  let len = String.length dir in
  List.map (fun s -> String.sub s (len + 1) (String.length s - len - 1)) l

let find_p dir            = find_ext dir ".p"
let find_hd dir           = find_ext dir ".hd"
let find_xml2 dir         = find_ext dir ".xml2"
let find_con_xml dir      = find_ext dir ".con.xml"
let find_ind_xml dir      = find_ext dir ".ind.xml"
let find_con_body_xml dir = find_ext dir ".con.body.xml"

