open Miz_parse

let escape_prime s =  Str.global_replace (Str.regexp "'") "\\'" s

(* now additonnally escape quotes *)
let escape_slash s =
  let s1 = Str.global_replace (Str.regexp "#") "#hash#" s in
  let s2 = Str.global_replace (Str.regexp "/") "#slash#" s1 in
  let s3 = Str.global_replace (Str.regexp "\"") "#quote#" s2 in
  let s4 = Str.global_replace (Str.regexp "\\") "#bslash#" s3 in
  s4

let is_lowercase c = 
  let i = Char.code c in i >= Char.code 'a' && i <= Char.code 'z' 
let is_uppercase c = 
  let i = Char.code c in i >= Char.code 'A' && i <= Char.code 'Z'
let is_numeric c =
  let i = Char.code c in i >= Char.code '0' && i <= Char.code '9'
let is_alphanumeric c =
  (is_lowercase c) or (is_uppercase c) or (is_numeric c) or 
  (Char.code c == Char.code '_')

let can f s = try (f s; true) with _ -> false

let is_tptp s =
  let f c = if is_alphanumeric c then () else raise Exit in
  can (String.iter f) s || List.mem s ["$true";"$t";"$o";"=>";"~"] 

let tptp_escape s =
  if is_tptp s then s else "'" ^ escape_prime (escape_slash s) ^ "'"


(* Read *)
let miz_read_prf_aux fname =
  let inc = Pervasives.open_in fname in
  let lexb = Lexing.from_channel inc in
  let rec prf () =
    try let v = try Miz_parse.hhtop Miz_lexer.hh2lex lexb
      with Parsing.YYexit a -> Obj.magic a in v :: (prf ())
    with End_of_file ->
      close_in inc;
      []
    | exn ->
      begin
        let curr = lexb.Lexing.lex_curr_p in
        let line = curr.Lexing.pos_lnum in
        let cnum = curr.Lexing.pos_cnum - curr.Lexing.pos_bol in
        Printf.eprintf "MIZ Parse error: %s line %i, char %i\n" fname line cnum;
        raise exn
      end
  in prf ()

let miz_read_prf dir filel = 
  List.map (fun fname -> (Filename.chop_extension fname, miz_read_prf_aux (dir ^ "/" ^ fname))) filel

let r2 = Str.regexp "[ \t]+"
let r3 = Str.regexp "^t.*$"

let miz_read_dir dir = miz_read_prf dir (Read.find_xml2 dir)

let mapped_names = Hashtbl.create 20000
let used_names = ref []
let const_names = Hashtbl.create 20000

let init_mapped_names file =
  let add fname _ s =
    match Str.split r2 s with
    | [a;b] -> Hashtbl.add mapped_names a b
    | _ -> failwith ("multiple names or empty line: " ^ s ^ ": " ^ fname)
  in
  Read.file_iter file (add file)

let variant_name s used =
  try
    let i = List.assoc s !used in 
    let rec name s i =
      let si = s ^ string_of_int i in
        if List.mem si (List.map fst !used) then name s (i + 1)
        else (used := (s,(i + 1)):: (si,0) :: !used; si)
    in 
      name s i 
  with Not_found -> (used := (s,0) :: !used; s)

let declare_perm c s =
  let s' = variant_name s used_names in
    Hashtbl.add const_names c s'; s'

(* Translate *)
let rec miz_to_hh miz_term = match miz_term with
  | Comb (Comb (Comb (Id "all",Varl l),x),y) ->
      Hh_parse.Comb (Hh_parse.Comb (Hh_parse.Id "all",mk_hh_abs l (miz_to_hh x)),
                     mk_hh_abs l (miz_to_hh y))
  | Comb (x,y) -> Hh_parse.Comb (miz_to_hh x,miz_to_hh y)
  | Abs (n,ty,tm) -> Hh_parse.Abs (n,miz_to_hh ty,miz_to_hh tm)
  | Id x          -> Hh_parse.Id x
  | Varl _        -> failwith "list of variables as argument"
and mk_hh_abs vl t = match vl with
    [] -> t
  | (n,ty) :: m -> Hh_parse.Abs(n,miz_to_hh ty,mk_hh_abs m t)

let rec change_names vl hh_term = match hh_term with
  | Hh_parse.Comb (x,y)    -> Hh_parse.Comb (change_names vl x,change_names vl y)
  | Hh_parse.Abs (n,ty,tm) -> Hh_parse.Abs (n,change_names vl ty,change_names (n :: vl) tm)
  | Hh_parse.Id x          -> if not (List.mem x vl) 
                              then 
                                try Hh_parse.Id (Hashtbl.find const_names x) with Not_found ->
                                let mx = (try Hashtbl.find mapped_names x with Not_found -> x) in
                                  Hh_parse.Id (declare_perm x mx)
                              else Hh_parse.Id x



(* Print *)
let rec strip_quant q (bvl,tm) = match tm with
    Hh_parse.Comb (Hh_parse.Id q, Hh_parse.Abs (n,ty,t)) -> strip_quant q (bvl @ [(n,ty)], t)
  | _ -> (bvl,tm)

let rec strip_abs (bvl,tm) = match tm with
    Hh_parse.Abs (n,ty,t) -> strip_abs (bvl @ [(n,ty)], t)
  | _ -> (bvl,tm)

let rec print_hh tm = match tm with
    Hh_parse.Comb (Hh_parse.Comb(Hh_parse.Id op,x),y) 
    when List.mem op ["&";"|";"=>";"=";">"] -> 
     "(" ^ print_hh x ^ " " ^ op ^ " " ^ print_hh y ^ ")"
  | Hh_parse.Comb (Hh_parse.Id q, Hh_parse.Abs _) 
    when List.mem q ["!";"?"] ->
      let (bvl,t) = strip_quant q ([],tm) in
       "(" ^ q ^ "[" ^ print_bvl bvl ^ "]: " ^ print_hh t ^ ")" 
  | Hh_parse.Comb (x,y) -> "(" ^ print_hh_comb tm ^ ")"
  | Hh_parse.Abs _ -> let (bvl,t) = strip_abs ([],tm) in
     "(^[" ^ print_bvl bvl ^ "]: " ^ print_hh t ^ ")"
  | Hh_parse.Id x -> if List.mem x [] then x else tptp_escape x
and print_hh_comb tm = match tm with
    Hh_parse.Comb(x,y) -> print_hh_comb x ^ " " ^ print_hh y
  | _                  -> print_hh tm
and print_bvl bvl = match bvl with
    [] -> failwith "empty list"
  | [(n,ty)] -> n ^ ": " ^ print_hh ty
  | (n,ty) :: m ->  n ^ ": " ^ print_hh ty ^ ", " ^ print_bvl m


let print_hh_line (a,b,c,d) = a ^ "(" ^ b ^ "," ^ c ^ "," ^ print_hh d ^ ")."

let writel filename strl =
  let oc = open_out filename in 
    List.iter (Printf.fprintf oc "%s\n") strl; 
  close_out oc

let write_hh_thy dir (thy,tml) = 
  writel (dir ^ "/" ^ thy ^ ".p") (List.map print_hh_line tml)  

let write_hh_thyl dir tmll = List.iter (write_hh_thy dir) tmll 

(* Main *)
let mk_dir s = try Unix.mkdir s 0o777 with Unix.Unix_error (Unix.EEXIST,_,_) -> ()

let clean_dir s = try Unix.mkdir s 0o777 
                  with Unix.Unix_error (Unix.EEXIST,_,_) -> 
                    ignore (Sys.command ("rm -rf " ^ s)); Unix.mkdir s 0o777

let foreach_map f tmll = List.map (fun (str,l) -> (str,List.map f l)) tmll

let init_miz_format =
  used_names := []; Hashtbl.clear mapped_names; Hashtbl.clear const_names

(* Extracting the types *)
let rec strip_forall tm = match tm with
    Hh_parse.Comb(Hh_parse.Id "!", Hh_parse.Abs (n,ty,t)) -> 
    let (bvl,t') = strip_forall t in
    (bvl @ [(n,ty)], t')
  | _ -> ([],tm)

let rec strip_comb tm = match tm with
    Hh_parse.Comb(f1,f2) -> let (oper,argl) = strip_comb f1 in
                            (oper, argl @ [f2])
  | Hh_parse.Id c -> (Hh_parse.Id c,[])
  | _             -> failwith "strip_comb"
 
let rec list_mk_tyimp tyl = match tyl with
    [] -> failwith "list_mk_tyimp"
  | [ty] -> ty
  | ty :: m ->
     Hh_parse.Comb(Hh_parse.Comb(Hh_parse.Id ">", ty),list_mk_tyimp m)
  
let dest_sort tm = 
  let (bvl,tm') = strip_forall tm in 
  let (tm2,tm3) = match strip_comb tm' with 
                   (Hh_parse.Id "sort",[t2;t3]) -> (t2,t3) 
                  | _           -> failwith ("dest_sort: " ^ print_hh tm)
  in
  let (c,_) = strip_comb tm2 in
  (c, list_mk_tyimp (List.map snd bvl @ [tm3]))

let dest_id tm = match tm with
    Hh_parse.Id c -> c
  | _ -> failwith "dest_id"

let rec contain_sort tm = match tm with
  | Hh_parse.Comb (x,y)   -> contain_sort x || contain_sort y
  | Hh_parse.Abs (n,ty,tm) -> contain_sort ty || contain_sort tm 
  | Hh_parse.Id x          -> x = "sort" || x = "sorts"

(* Printing theorems and (types) *)
let miz_format dir_in file_const dir_out =
  init_miz_format;
  init_mapped_names file_const;
  let tmll = miz_read_dir dir_in in
  let f (_,x,y,z) =
    if String.get x 0 = 't' || String.get x 0 = 'd'
    then 
      let tm = change_names [] (miz_to_hh z) in
      if contain_sort tm 
      then []
      else [("tt",x,"ax",tm)]
    (* else if String.get x 0 = 'd' && String.get x 1 = 't'
    then 
      let tm = miz_to_hh z in
      try 
        let (c,ty) = dest_sort tm in
        if String.get (dest_id c) 0 = 'c' then []
        else
          let new_c = change_names [] c in
          let new_ty = change_names [] ty in
          [("tt",dest_id new_c,"ty",new_ty)]
      with _ -> (errorl := print_hh tm :: !errorl ; [])
    *)
    else []
  in
  let tmll = foreach_map f tmll in
  let tmll = List.map (fun (x,y)-> (x,List.concat y)) tmll in
    ignore (clean_dir dir_out); write_hh_thyl dir_out tmll 

