(*-------------------------------------------------------------------------- *)
(* Create a set of properties of each constant                               *)
(*-------------------------------------------------------------------------- *)

open Toolbox
open Init
open Hh_term

(*--------------------------------------------------------------------------
  Options (used for initial matching of Mizar and HOL4)
  Chose about the double of the number of constants used
  -------------------------------------------------------------------------- *)

let no_type_flag = ref false
let simple_type_flag = ref false
let constant_type_flag = ref false
let cnfcom_flag = ref true
let max_constants = 30000
let max_thms = 100000
let no_congruence_flag = ref false
let no_subterm_congruence_flag = ref false

(*--------------------------------------------------------------------------
  Property types
  -------------------------------------------------------------------------- *)

let wildcard_cint = -1
let this_cint = -2

module Prop = struct
  type t = int * int list
  let compare a b = compare a b
  let rep_of (n,cl) =
    let map_c c = if c < 0 then c else wildcard_cint in
    (n,List.map map_c cl)
  let compare_rep a b =
    compare (rep_of a) (rep_of b)
end

module RedRes = struct
  type t = (int * int)
  let compare a b = compare a b
end

(* allows to share a constant between properties *)
module PropRef = struct
  type t = int * int ref ref list
  let to_prop (a,al) = (a, List.map (fun x -> !(!x)) al)
  let rep_of p = Prop.rep_of (to_prop p)
  let compare p1 p2 = Prop.compare (to_prop p1) (to_prop p2)
  let compare_rep p1 p2 = Prop.compare_rep (to_prop p1) (to_prop p2)
  
end

module Intaux = struct
  type t = int
  let compare a b = compare a b
end

module StrMap = Map.Make(String)
module StrSet = Set.Make(String)
module IntMap = Map.Make(Intaux)
module PropMap = Map.Make(Prop)
module IntSet = Set.Make(Intaux)
module PropSet = Set.Make(Prop)
module Subst = Set.Make(RedRes)

(* to be remove *)
let mk_subst l = List.fold_right Subst.add l Subst.empty

let symmetric_of sb =
  let l = ref [] in
  let f (a,b) = l := (b,a) :: !l in
  Subst.iter f sb;
  mk_subst (!l)

let are_symmetric sb1 sb2 = (symmetric_of sb1 = sb2)

(*--------------------------------------------------------------------------
  Syntax
  -------------------------------------------------------------------------- *)

type tm_t = | V of (int * tm_t)
            | C of (int * tm_t)
            | D of string
            | App  of  (tm_t * tm_t)
            | Lambda of (tm_t * tm_t)
            | Forall of (tm_t * tm_t)
            | Exists of (tm_t * tm_t)
            | And  of (tm_t * tm_t)
            | Or of (tm_t * tm_t)
            | Imply  of (tm_t * tm_t)
            | Not of tm_t
            | Eq of (tm_t * tm_t)

let compare_cpl comp1 comp2 (a,b) (c,d) = 
  let n = comp1 a c in
  if n = 0 then comp2 b d else n

let rec compare_tm tm1 tm2 = match (tm1,tm2) with
    (V(v1,ty1),V(v2,ty2)) -> compare_cpl compare compare_tm (v1,ty1) (v2,ty2)
  | (V(_,_),_)            -> -1
  | (_,V(_,_))            -> 1
  | (C(c1,ty1),C(c2,ty2)) -> compare_cpl compare compare_tm (c1,ty1) (c2,ty2)
  | (C(_,_),_)            -> -1
  | (_,C(_,_))            -> 1
  | (D(d1),D(d2))         -> compare d1 d2
  | (D(_),_)              -> -1
  | (_,D(_))              -> 1
  | (App(fa1,fb1),App(fa2,fb2)) -> 
      compare_cpl compare_tm compare_tm (fa1,fb1) (fa2,fb2)
  | (App(_,_),_)          -> -1
  | (_,App(_,_))          -> 1
  | (Lambda(va,fa),Lambda(vb,fb)) -> 
      compare_cpl compare_tm compare_tm (va,fa) (vb,fb)
  | (Lambda(_,_),_)          -> -1
  | (_,Lambda(_,_))          -> 1
  | (Forall(va,fa),Forall(vb,fb)) -> 
      compare_cpl compare_tm compare_tm (va,fa) (vb,fb)
  | (Forall(_,_),_)          -> -1
  | (_,Forall(_,_))          -> 1
  | (Exists(va,fa),Exists(vb,fb)) -> 
      compare_cpl compare_tm compare_tm (va,fa) (vb,fb)
  | (Exists(_,_),_)          -> -1
  | (_,Exists(_,_))          -> 1
  | (And(fa1,fb1),And(fa2,fb2)) -> 
      compare_cpl compare_tm compare_tm (fa1,fb1) (fa2,fb2)
  | (And(_,_),_)          -> -1
  | (_,And(_,_))          -> 1
  | (Or(fa1,fb1),Or(fa2,fb2)) -> 
      compare_cpl compare_tm compare_tm (fa1,fb1) (fa2,fb2)
  | (Or(_,_),_)          -> -1
  | (_,Or(_,_))          -> 1
  | (Imply(fa1,fb1),Imply(fa2,fb2)) -> 
      compare_cpl compare_tm compare_tm (fa1,fb1) (fa2,fb2)
  | (Imply(_,_),_)          -> -1
  | (_,Imply(_,_))          -> 1
  | (Not(f1),Not(f2)) -> compare_tm f1 f2
  | (Not(_),_)          -> -1
  | (_,Not(_))          -> 1
  | (Eq(fa1,fb1),Eq(fa2,fb2)) -> 
      compare_cpl compare_tm compare_tm (fa1,fb1) (fa2,fb2)

(* Compare terms with position of variables and constants taken into consideration 
   (position in the other term should also be used)
*)

module Tm = struct
  type t = tm_t
  let compare tm1 tm2 = compare_tm tm1 tm2  
end

module TmMap = Map.Make(Tm)
module TmSet = Set.Make(Tm)

let find_int dict n key =
  try TmMap.find key (!dict) with Not_found ->
   let m = !n in
     dict := TmMap.add key m (!dict);
     n := (!n) + 1; m

let find_only_int dict n key =
  try Some (TmMap.find key (!dict)) with _ -> None

let reset_dict dict n = (dict := TmMap.empty; n := 0)

let cp_var1_dict = ref TmMap.empty		
let cp_var1_n = ref 0
let cp_const1_dict = ref TmMap.empty		
let cp_const1_n = ref 0

let cp_var2_dict = ref TmMap.empty		
let cp_var2_n = ref 0
let cp_const2_dict = ref TmMap.empty		
let cp_const2_n = ref 0

(* Warning: 
AC is not handle correctly should be on whole terms and not on the arguments, 
but in this way it is still correct for commutativity 
and faster at handling associativity (albeit not fully correct) 
*)

(* Doesn't do alpha conversion in types (would only be needed for Coq-Matita) *)
let rec compare_tm_abstract tm1 tm2 = match (tm1,tm2) with
    (V(v1,ty1),V(v2,ty2)) -> 
    let (indice_v1,indice_v2) =
      (find_int cp_var1_dict cp_var1_n (V(v1,ty1)),
       find_int cp_var2_dict cp_var2_n (V(v2,ty2)))
    in 
    let (other_v1,other_v2) =
      (find_only_int cp_var2_dict cp_var2_n (V(v1,ty1)),
       find_only_int cp_var1_dict cp_var1_n (V(v2,ty2)))
    in
    compare_cpl compare compare (indice_v1,other_v1) (indice_v2,other_v2)
  | (V(_,_),_)            -> -1
  | (_,V(_,_))            -> 1
  | (C(c1,ty1),C(c2,ty2)) -> 
    let (indice_c1,indice_c2) = 
      (find_int cp_const1_dict cp_const1_n (C(c1,ty1)),
       find_int cp_const2_dict cp_const2_n (C(c2,ty2)))
    in 
    let (other_c1,other_c2) =
      (find_only_int cp_const2_dict cp_const2_n (C(c1,ty1)),
       find_only_int cp_const1_dict cp_const1_n (C(c2,ty2)))
    in
    compare_cpl compare compare (indice_c1,other_c1) (indice_c2,other_c2)
  | (C(_,_),_)            -> -1
  | (_,C(_,_))            -> 1
  | (D(d1),D(d2))         -> compare d1 d2
  | (D(_),_)              -> -1
  | (_,D(_))              -> 1
  | (App(fa1,fb1),App(fa2,fb2)) -> 
      compare_cpl compare_tm_abstract compare_tm_abstract (fa1,fb1) (fa2,fb2)
  | (App(_,_),_)          -> -1
  | (_,App(_,_))          -> 1
  | (Lambda(va,fa),Lambda(vb,fb)) -> 
      compare_cpl compare_tm_abstract compare_tm_abstract (va,fa) (vb,fb)
  | (Lambda(_,_),_)          -> -1
  | (_,Lambda(_,_))          -> 1
  | (Forall(va,fa),Forall(vb,fb)) -> 
      compare_cpl compare_tm_abstract compare_tm_abstract (va,fa) (vb,fb)
  | (Forall(_,_),_)          -> -1
  | (_,Forall(_,_))          -> 1
  | (Exists(va,fa),Exists(vb,fb)) -> 
      compare_cpl compare_tm_abstract compare_tm_abstract (va,fa) (vb,fb)
  | (Exists(_,_),_)          -> -1
  | (_,Exists(_,_))          -> 1
  | (And(fa1,fb1),And(fa2,fb2)) -> 
      compare_cpl compare_tm_abstract compare_tm_abstract (fa1,fb1) (fa2,fb2)
  | (And(_,_),_)          -> -1
  | (_,And(_,_))          -> 1
  | (Or(fa1,fb1),Or(fa2,fb2)) -> 
      compare_cpl compare_tm_abstract compare_tm_abstract (fa1,fb1) (fa2,fb2)
  | (Or(_,_),_)          -> -1
  | (_,Or(_,_))          -> 1
  | (Imply(fa1,fb1),Imply(fa2,fb2)) -> 
      compare_cpl compare_tm_abstract compare_tm_abstract (fa1,fb1) (fa2,fb2)
  | (Imply(_,_),_)          -> -1
  | (_,Imply(_,_))          -> 1
  | (Not(f1),Not(f2)) -> compare_tm_abstract f1 f2
  | (Not(_),_)          -> -1
  | (_,Not(_))          -> 1
  | (Eq(fa1,fb1),Eq(fa2,fb2)) -> 
      compare_cpl compare_tm_abstract compare_tm_abstract (fa1,fb1) (fa2,fb2)

let compare_tm_abs tm1 tm2 = 
  reset_dict cp_var1_dict cp_var1_n;
  reset_dict cp_var2_dict cp_var2_n;
  reset_dict cp_const1_dict cp_const1_n;
  reset_dict cp_const2_dict cp_const2_n;
  compare_tm_abstract tm1 tm2


(*--------------------------------------------------------------------------
  Global reference for handling 2 provers
  -------------------------------------------------------------------------- *)

let prover_ref = ref 0
let set_prover i = prover_ref := i 

(*--------------------------------------------------------------------------
  Handling warnings (non critical errors)
  -------------------------------------------------------------------------- *)

let declaration_errorl1 = ref []
let declaration_errorl2 = ref []
let inference_errorl1 = ref []
let inference_errorl2 = ref []

let init_errorl () =
  declaration_errorl1 := [];
  declaration_errorl2 := [];
  inference_errorl1 := [];
  inference_errorl2 := []

let declaration_errorl () = match !prover_ref with
    1 -> declaration_errorl1
  | 2 -> declaration_errorl2
  | _ -> failwith "no prover_ref specified"

let inference_errorl () = match !prover_ref with
    1 -> inference_errorl1
  | 2 -> inference_errorl2
  | _ -> failwith "no prover_ref specified"

(*--------------------------------------------------------------------------
  Array of pointed constants
  -------------------------------------------------------------------------- *)

(* List containing constants which properties are pointing to *)
(* Each array represent a substitution from the original constant to the 
   mapped constant.*)
let constant_array1 = Array.init max_constants ref
let constant_array2 = Array.init max_constants ref
let nb_of_constants1 = ref 0
let nb_of_constants2 = ref 0

let array_copy a1 a2 =
  for i = 0 to max_constants - 1 do
    a2.(i) <- a1.(i)
  done

let init_array () =
  let ca1 = Array.init max_constants ref in
  let ca2 = Array.init max_constants ref in
  nb_of_constants1 := 0;
  nb_of_constants2 := 0;
  array_copy ca1 constant_array1;
  array_copy ca2 constant_array2;
  prover_ref := 0

let nb_of_constants () = match !prover_ref with
    1 -> nb_of_constants1
  | 2 -> nb_of_constants2
  | _ -> failwith "no prover_ref specified"

let constant_array () = match !prover_ref with
    1 -> constant_array1
  | 2 -> constant_array2
  | _ -> failwith "no prover_ref specified"

(* Constant informations *)
let constant_hash1 = Hashtbl.create max_constants
let constant_hash2 = Hashtbl.create max_constants
let inv_constant_hash1 = Hashtbl.create max_constants
let inv_constant_hash2 = Hashtbl.create max_constants
let constant_counter = ref 0

let init_constant_hash () =
  Hashtbl.clear constant_hash1;
  Hashtbl.clear inv_constant_hash1;
  Hashtbl.clear constant_hash2;
  Hashtbl.clear inv_constant_hash2 

let constant_hash () = match !prover_ref with
    1 -> constant_hash1
  | 2 -> constant_hash2
  | _ -> failwith "no prover_ref specified"

let inv_constant_hash () = match !prover_ref with
    1 -> inv_constant_hash1
  | 2 -> inv_constant_hash2
  | _ -> failwith "no prover_ref specified"


let cint_of_constant ch s = try fst (Hashtbl.find ch s) 
  with _ -> failwith ("cint_of_constant: " ^ s)
let type_of_constant ch s = try snd (Hashtbl.find ch s) 
  with _ -> failwith ("type_of_constant: " ^ s)
let both_of_constant ch s = try Hashtbl.find ch s
  with _ -> failwith ("both_of_constant: " ^ s)

let name_of_cint ich n = try fst (Hashtbl.find ich n)
  with _ -> failwith ("name_of_cint: " ^ string_of_int n)
let type_of_cint ich n = try snd (Hashtbl.find ich n)
  with _ -> failwith ("type_of_cint: " ^ string_of_int n)

(*--------------------------------------------------------------------------
  Converting hh matching term to string
  -------------------------------------------------------------------------- *)

let rec hhm_to_string tm =
  match tm with
    V(v,ty)  -> if !no_type_flag 
                then "V" ^ string_of_int v
                else "(" ^ "V" ^ string_of_int v ^ ":" ^ hhm_to_string ty ^ ")"
  | C(c,ty)  -> if !no_type_flag 
                then "C" ^ string_of_int c
                else "(" ^ name_of_cint (inv_constant_hash ()) c ^ ":" ^ hhm_to_string ty ^ ")" 
  | D(d)     ->  "D" ^ d
  | App(f1,f2)   -> "(" ^ hhm_to_string f1 ^ " " ^ hhm_to_string f2 ^ ")"
  | Lambda(x,f)  -> "(^ " ^ hhm_to_string x ^ " , " ^ hhm_to_string f ^ ")"
  | Forall(x,f)  -> "(! " ^ hhm_to_string x ^ " , " ^ hhm_to_string f ^ ")"
  | Exists(x,f)  -> "(? " ^ hhm_to_string x ^ " , " ^ hhm_to_string f ^ ")"
  | And(f1,f2)   -> "(& " ^ hhm_to_string f1 ^ " " ^ hhm_to_string f2 ^ ")"
  | Or(f1,f2)    -> "(| " ^ hhm_to_string f1 ^ " " ^ hhm_to_string f2 ^ ")"
  | Imply(f1,f2) -> "(=> " ^ hhm_to_string f1 ^ " " ^ hhm_to_string f2 ^ ")"
  | Not(f)       -> "(~ " ^ hhm_to_string f ^ ")"
  | Eq(f1,f2)    -> "(= " ^ hhm_to_string f1 ^ " " ^ hhm_to_string f2 ^ ")"

let rec hhm_to_string_notype tm =
  match tm with
    V(v,ty)  -> "V" ^ string_of_int v
  | C(c,ty)  -> name_of_cint (inv_constant_hash ()) c
  | D(d)     ->  "D" ^ d
  | App(f1,f2)   -> "(" ^ hhm_to_string_notype f1 ^ " " ^ hhm_to_string_notype f2 ^ ")"
  | Lambda(x,f)  -> "(^ " ^ hhm_to_string_notype x ^ " , " ^ hhm_to_string_notype f ^ ")"
  | Forall(x,f)  -> "(! " ^ hhm_to_string_notype x ^ " , " ^ hhm_to_string_notype f ^ ")"
  | Exists(x,f)  -> "(? " ^ hhm_to_string_notype x ^ " , " ^ hhm_to_string_notype f ^ ")"
  | And(f1,f2)   -> "(" ^ hhm_to_string_notype f1 ^ " & " ^ hhm_to_string_notype f2 ^ ")"
  | Or(f1,f2)    -> "(" ^ hhm_to_string_notype f1 ^ " | " ^ hhm_to_string_notype f2 ^ ")"
  | Imply(f1,f2) -> "(" ^ hhm_to_string_notype f1 ^ " => " ^ hhm_to_string_notype f2 ^ ")"
  | Not(f)       -> "(~ " ^ hhm_to_string_notype f ^ ")"
  | Eq(f1,f2)    -> "(" ^ hhm_to_string_notype f1 ^ " = " ^ hhm_to_string_notype f2 ^ ")"



let lisp_of l = "(" ^ String.concat " " l ^ ")" 

let rec hhm_to_lisp tm = match tm with
    V(v,ty)  -> if !no_type_flag
                then  "V" ^ string_of_int v
                else lisp_of ["\":\""; "V" ^ string_of_int v; hhm_to_lisp ty]
  | C(c,ty)  -> if !no_type_flag 
                then "C" ^ string_of_int c
                else lisp_of ["\":\""; "C" ^ string_of_int c; hhm_to_lisp ty]
  | D(d)     -> "\"" ^ d ^ "\""
  | App(f1,f2)   -> lisp_of [hhm_to_lisp f1; hhm_to_lisp f2]
  | Lambda(x,f)  -> lisp_of ["\"^\""; hhm_to_lisp x; hhm_to_lisp f]
  | Forall(x,f)  -> lisp_of ["\"!\""; hhm_to_lisp x; hhm_to_lisp f]
  | Exists(x,f)  -> lisp_of ["\"?\""; hhm_to_lisp x; hhm_to_lisp f]
  | And(f1,f2)   -> lisp_of ["\"&\""; hhm_to_lisp f1; hhm_to_lisp f2]
  | Or(f1,f2)    -> lisp_of ["\"|\""; hhm_to_lisp f1; hhm_to_lisp f2]
  | Imply(f1,f2) -> lisp_of ["\"=>\""; hhm_to_lisp f1; hhm_to_lisp f2]
  | Not(f)       -> lisp_of ["\"~\""; hhm_to_lisp f]
  | Eq(f1,f2)    -> lisp_of ["\"=\""; hhm_to_lisp f1; hhm_to_lisp f2]
  


(* For subterms : variables do appear in types of subterms but alpha conversion
   is not done yet so they are just zeroed.*)
let rec hhm_to_string_novar tm = 
  let ich = inv_constant_hash () in
  match tm with
    V(v,ty)  -> "$V_" ^ (hhm_to_string_novar ty)
  | C(c,ty)  -> name_of_cint ich c
  | D(d)     -> d
  | App(f1,f2) -> 
    "(" ^ hhm_to_string_novar f1 ^ " " ^ hhm_to_string_novar f2 ^ ")"
  | Lambda(x,f)  -> 
    "(^ " ^ hhm_to_string_novar x ^ ", " ^ hhm_to_string_novar f ^ ")"
  | Forall(x,f)  -> 
    "(! " ^ hhm_to_string_novar x ^ ", " ^ hhm_to_string_novar f ^ ")"
  | Exists(x,f)  -> 
    "(? " ^ hhm_to_string_novar x ^ ", " ^ hhm_to_string_novar f ^ ")"
  | And(f1,f2)   -> 
    "(& " ^ hhm_to_string_novar f1 ^ " " ^ hhm_to_string_novar f2 ^ ")"
  | Or(f1,f2)    -> 
    "(| " ^ hhm_to_string_novar f1 ^ " " ^ hhm_to_string_novar f2 ^ ")"
  | Imply(f1,f2) -> 
    "(=> " ^ hhm_to_string_novar f1 ^ " " ^ hhm_to_string_novar f2 ^ ")"
  | Not(f)       -> 
    "(~ " ^ hhm_to_string_novar f ^ ")"
  | Eq(f1,f2)    -> 
    "(= " ^ hhm_to_string_novar f1 ^ " " ^ hhm_to_string_novar f2 ^ ")"

(*--------------------------------------------------------------------------
  Fof printer
  -------------------------------------------------------------------------- *)

let rec app_arguments formula = match formula with
    App (f1,f2) -> let (oper,argl) = app_arguments f1 in 
                           (oper, (argl @ [f2]))
  | _          -> (formula,[])

let rec subterm_to_fof formula = 
  match formula with
    C(c,ty)  -> name_of_cint (inv_constant_hash ()) c
  | App(f1,f2) -> 
     let (oper, argl) = app_arguments formula in
       subterm_to_fof oper ^ 
         "(" ^ String.concat "," (List.map subterm_to_fof argl) ^ ")"
  | _ -> failwith "not valid subterm"

(*--------------------------------------------------------------------------
  Subterm printer
  -------------------------------------------------------------------------- *)

let string_of_subterm formula = 
  if !fof_flag 
    then subterm_to_fof formula
    else hhm_to_string_notype formula

let rec app_arguments formula = match formula with
    App (f1,f2) -> let (oper,argl) = app_arguments f1 in 
                           (oper, (argl @ [f2]))
  | _          -> (formula,[])

let rec subterm_to_fof formula = 
  match formula with
    C(c,ty)  -> name_of_cint (inv_constant_hash ()) c
  | App(f1,f2) -> 
     let (oper, argl) = app_arguments formula in
       subterm_to_fof oper ^ 
         "(" ^ String.concat "," (List.map subterm_to_fof argl) ^ ")"
  | _ -> failwith "not valid subterm"

(*--------------------------------------------------------------------------
  Initialization
  -------------------------------------------------------------------------- *)

(* Constant declaration *)
let declare_constant c ty =
  let ch = constant_hash () in
  let ich = inv_constant_hash () in
  let n = !constant_counter in
  if Hashtbl.mem ch c then () else
  begin
    Hashtbl.replace ch c (n,ty);
    Hashtbl.replace ich n (c,ty);
    incr constant_counter
  end

(* Format a term *)
let rm_bool_flag = ref false

let fixed_constants_wobool () = 
  if !rm_bool_flag then list_diff fixed_constants ["$o"] else fixed_constants

let rec init_fmt vl f t = match t with
  | Comb (x,y)    -> App (init_fmt vl f x,init_fmt vl f y)
  | Abs (v,ty,tm) -> 
      let n = f v in
      let ty1 = 
        if !no_type_flag then D("$true") else init_fmt vl f ty
      in   
      let tm1 = init_fmt ((v,ty1) :: vl) f tm in
      Lambda(V(n,ty1),tm1)  
  | Id id          -> 
      if !constant_type_flag then
        begin                      
        if List.mem id (List.map fst vl) 
          then V(f id,List.assoc id vl)
        else if List.mem id (fixed_constants_wobool ())
          then D(id)
          else try C(both_of_constant (constant_hash ()) id) with _ ->
          begin
          let lref = declaration_errorl () in
          lref := id :: !lref;
          declare_constant id (D("$true"));
          C(both_of_constant (constant_hash ()) id)
          end        
        end
      else
        begin
        if List.mem id (List.map fst vl)
          then V(f id,List.assoc id vl)
        else if List.mem id (fixed_constants_wobool ())
          then D(id)
        else 
          try C(both_of_constant (constant_hash ()) id) with _ ->
          begin 
          declare_constant id (D("$true"));
          C(both_of_constant (constant_hash ()) id)
          end
        end

let format t = 
  let hvar = Hashtbl.create 100 in
  let hvarn = ref 0 in
  let find_hvar s = try Hashtbl.find hvar s with _ -> 
                    let n = !hvarn in 
                    incr hvarn; Hashtbl.add hvar s n; n
  in
  init_fmt [] find_hvar t

let declare_constl l =
  let f (c,hh_ty) =
    if !constant_type_flag then
      let ty = format hh_ty in
      declare_constant c ty
    else declare_constant c (D("$true"))
  in
  List.iter f l

let rec fmt_tm tm = match tm with
  | V(v,ty)      -> V(v,fmt_tm ty)
  | C(c,ty)      -> C(c,fmt_tm ty)
  | D(c)         -> D(c)
  | App(D("!"),Lambda(x,f))  -> Forall(fmt_tm x,fmt_tm f) 
  | App(D("?"),Lambda(x,f))  -> Exists(fmt_tm x,fmt_tm f) 
  | App(App(D("&"),f1),f2) -> And(fmt_tm f1,fmt_tm f2)
  | App(App(D("|"),f1),f2) -> Or(fmt_tm f1,fmt_tm f2)
  | App(App(D("=>"),f1),f2) -> Imply(fmt_tm f1,fmt_tm f2)
  | App(D("~"),f)            -> Not(fmt_tm f)
  | App(App(D("="),f1),f2)   -> Eq(fmt_tm f1,fmt_tm f2)
  | App(App(D("<=>"),f1),f2)   -> Eq(fmt_tm f1,fmt_tm f2)
  | Lambda(x,f)                -> Lambda(fmt_tm x,fmt_tm f)
  | App(f1,f2)                 -> App(fmt_tm f1,fmt_tm f2)
  | _            -> failwith "fmt_tm"

let rec sub_tm change tm = match tm with
  | V(v,ty)      -> V(v,ty)
  | C(c,ty)      -> C(c,ty)
  | D(d)         -> D(d)
  | App(f1,f2)   -> App(change f1, change f2)
  | Lambda(x,f)  -> Lambda(change x, change f)
  | Forall(x,f)  -> Forall(change x, change f)
  | Exists(x,f)  -> Exists(change x, change f)
  | And(f1,f2)   -> And(change f1, change f2)
  | Or(f1,f2)    -> Or(change f1, change f2)
  | Imply(f1,f2) -> Imply(change f1, change f2)
  | Not(f)       -> Not(change f)
  | Eq(f1,f2)    -> Eq(change f1, change f2)

let rec sub_tm_deep change tm = match tm with
  | V(v,ty)      -> V(v,change ty)
  | C(c,ty)      -> C(c,change ty)
  | D(d)         -> D(d)
  | App(f1,f2)   -> App(change f1, change f2)
  | Lambda(x,f)  -> Lambda(change x, change f)
  | Forall(x,f)  -> Forall(change x, change f)
  | Exists(x,f)  -> Exists(change x, change f)
  | And(f1,f2)   -> And(change f1, change f2)
  | Or(f1,f2)    -> Or(change f1, change f2)
  | Imply(f1,f2) -> Imply(change f1, change f2)
  | Not(f)       -> Not(change f)
  | Eq(f1,f2)    -> Eq(change f1, change f2)


let cint_of_hhm tm =
  let rl = ref [] in
  let rec get_constantl_aux t = match t with
    | C(c,ty)     -> if not (List.mem c (!rl)) then rl := (c :: !rl) else ();
                     C(c,get_constantl_aux ty) 
    | _           -> sub_tm_deep get_constantl_aux t
  in
    (ignore (get_constantl_aux tm); !rl)

let total_of_constant tm = List.length (mk_set (cint_of_hhm tm))

(*--------------------------------------------------------------------------
  Normalization (not deep)
  -------------------------------------------------------------------------- *)

let norm_steps_glob = ref 0
let norm_steps_warning = ref 0

let rec remove_imply tm =
  match tm with
  | Imply(f1,f2) -> Or(Not(remove_imply f1), remove_imply f2)
  | _            -> sub_tm remove_imply tm

let rec move_neg_in tm =
  match tm with
  | Not(Forall(v,f)) -> Exists(v,move_neg_in (Not(f)))
  | Not(Exists(v,f)) -> Forall(v,move_neg_in (Not(f)))
  | Not(And(f1,f2))  -> Or(move_neg_in (Not(f1)), move_neg_in (Not(f2)))
  | Not(Or(f1,f2))   -> And(move_neg_in (Not(f1)), move_neg_in (Not(f2)))
  | Not(Not(f1))     -> move_neg_in f1
  | _                -> sub_tm move_neg_in tm

let rec move_forall_out tm =
  match tm with
  | And(Forall(x,f1),f2) -> Forall(x, move_forall_out (And(f1,f2)))
  | And(f1,Forall(x,f2)) -> Forall(x, move_forall_out (And(f1,f2)))
  | Or(Forall(x,f1),f2) -> Forall(x, move_forall_out (Or(f1,f2)))
  | Or(f1,Forall(x,f2)) -> Forall(x, move_forall_out (Or(f1,f2)))
  | _                   -> sub_tm move_forall_out tm
			
let rec distrib_disj tm =
  if !norm_steps_glob > 1000 then (incr norm_steps_warning; tm) 
  else
  begin
  match tm with
  | Or(f1,And(f2,f3)) -> 
    (incr norm_steps_glob; And(distrib_disj (Or(f1,f2)), distrib_disj (Or(f1,f3))))
  | Or(And(f1,f2),f3) -> 
    (incr norm_steps_glob; And(distrib_disj (Or(f1,f3)), distrib_disj (Or(f2,f3))))
  | _                 -> sub_tm distrib_disj tm
  end
 
let normalize tm = 
  norm_steps_glob := 0;
  distrib_disj (move_forall_out (move_neg_in (remove_imply tm)))

(*--------------------------------------------------------------------------
  Reduce to simple types: (for Mizar) to be used with constant_type_flag off.
  -------------------------------------------------------------------------- *)

(* allow a lot more mapping and especially mapping with $true = set in 
mizar *)
let rec simplify_type tm = match tm with
  | V(x,ty) -> let new_ty = 
                 match ty with
                   D(id)  -> declare_constant id (D("$true")); 
                             C(both_of_constant (constant_hash ()) id)
                 | _       -> let id = ("$ " ^ hhm_to_string_novar ty) in
                              declare_constant id (D("$true"));
                              C(both_of_constant (constant_hash ()) id)
               in V(x,new_ty)
  | _       -> sub_tm simplify_type tm


(*--------------------------------------------------------------------------
  Commutativity and associativity
  -------------------------------------------------------------------------- *)

let commute_flag = ref false

(* quantifiers *)
let rec forall_arguments tm = match tm with
  | Forall(x,f) -> let (vl,f') = forall_arguments f in (x :: vl, f')
  | _           -> ([],tm)

let rec exists_arguments tm = match tm with
  | Exists(x,f) -> let (vl,f') = exists_arguments f in (x :: vl, f')
  | _           -> ([],tm)

let rec lambda_arguments tm = match tm with
  | Lambda(x,f) -> let (vl,f') = lambda_arguments f in (x :: vl, f')
  | _           -> ([],tm)

let rec mk_forall_list (vl,f) = match vl with
  | []     -> f
  | v :: m -> Forall(v, mk_forall_list (m,f))

let rec mk_exists_list (vl,f) = match vl with
  | []     -> f
  | v :: m -> Exists(v, mk_exists_list (m,f))

let rec mk_lambda_list (vl,f) = match vl with
  | []     -> f
  | v :: m -> Lambda(v, mk_lambda_list (m,f))

(* logical operators *)
let rec or_arguments tm = match tm with
  | Or(f1,f2) -> or_arguments f1 @ or_arguments f2
  | _         -> [tm]

let rec and_arguments tm = match tm with
  | And(f1,f2) -> and_arguments f1 @ and_arguments f2
  | _         -> [tm]

let rec mk_or_list tml = match tml with
  | []   -> failwith "mk_or_list" 
  | [tm] -> tm
  | tm :: m -> Or(tm, mk_or_list m) 

let rec mk_and_list tml = match tml with
  | []   -> failwith "mk_and_list" 
  | [tm] -> tm
  | tm :: m -> And(tm, mk_and_list m) 

let eq_arguments tm = match tm with
  | Eq(a,b) -> [a;b]
  | _       -> failwith "eq_arguments"

let mk_eq tm = match tm with
  | [a;b]   -> Eq(a,b)
  | _       -> failwith "eq_arguments"


(* functions *)
let rec fun_arguments tm = match tm with
  | App(f1,f2) -> let (oper,argl) = fun_arguments f1 in 
                  (oper, (argl @ [f2]))
  | _          -> (tm,[])

let rec mk_fun_list (oper,argl) = match argl with
  | []      -> oper
  | tm :: m -> mk_fun_list (App(oper,tm), m)


(* bottom up: commute inside subterms before comparing them *)

let rec first_appear x l n = match l with
    [] -> n
  | a :: m -> if x = a then n else first_appear x m (n+1)

let compare_appear l x1 x2  =
  compare (first_appear x1 l 0) (first_appear x2 l 0)

let compare_tm_abs_fst (tm1,_) (tm2,_) =
  compare_tm_abs tm1 tm2

(* ordering variables from left to right *)
let rec commute tm = match tm with
  | V(v,ty)      -> let (ty',vl') = commute ty in
                    (V(v,ty'), V(v,ty) :: vl')
  | C(c,ty)      -> let (ty',vl') = commute ty in
                    (C(c,ty'), vl')
  | D(d)         -> (D(d),[])
  | Lambda(x,f)  -> commute_quant lambda_arguments mk_lambda_list tm
  | Forall(x,f)  -> commute_quant forall_arguments mk_forall_list tm
  | Exists(x,f)  -> commute_quant exists_arguments mk_exists_list tm
  | And(f1,f2)   -> commute_oper and_arguments mk_and_list tm
  | Or(f1,f2)    -> commute_oper or_arguments mk_or_list tm
  | Eq(f1,f2)    -> commute_oper eq_arguments mk_eq tm
  | App(f1,f2)   -> if !commute_flag 
                    then
                      let (oper,argl) = fun_arguments tm in
                      let sortedl = commute_then_sort argl in
                      let tml,vll = List.map fst sortedl, List.map snd sortedl in 
                      (mk_fun_list (oper,tml), List.concat vll)
                    else
                    let (f1',vl1'), (f2',vl2') = commute f1, commute f2 in
                    (App(f1',f2'), vl1' @ vl2')
  | Imply(f1,f2) -> let (f1',vl1'), (f2',vl2') = commute f1, commute f2 in
                    (Imply(f1',f2'), vl1' @ vl2')
  | Not(f)       -> let (f',vl') = commute f in (Not(f'), vl')
                     

and commute_then_sort l = List.sort compare_tm_abs_fst (List.map commute l)
and commute_list l = List.map commute l

and commute_quant quant_arguments mk_quant tm =
  let (bvl,f') = quant_arguments tm in
  let (f'',vl) = commute f' in
  let bvl' = List.sort (compare_appear vl) bvl in
  (mk_quant (bvl', f''), bvl' @ vl)
and commute_oper oper_arguments mk_oper tm =
  let l = oper_arguments tm in
  let sortedl = commute_then_sort l in
  let tml,vll = List.map fst sortedl, List.map snd sortedl in 
  (mk_oper tml, List.concat vll)

(*--------------------------------------------------------------------------
  Find subterms
  -------------------------------------------------------------------------- *)

let subterm_flag = ref false
let norm_hash = Hashtbl.create max_thms

(* filter constants by default *)
let rec all_subtm tm = match tm with
  | V(v,ty)      -> (fst (all_subtm ty),false)
  | C(c,ty)      -> (fst (all_subtm ty),true)
  | D(d)         -> (TmSet.empty,false)
  | App(f1,f2)   -> let ((s1,b1),(s2,b2)) = (all_subtm f1, all_subtm f2) in
                    let b = b1 && b2 in
                      if b 
                      then (TmSet.add tm (TmSet.union s1 s2),b)
                      else (TmSet.union s1 s2,b)
  | Lambda(x,f)
  | Forall(x,f)
  | Exists(x,f)  -> 
     (TmSet.union (fst (all_subtm x)) (fst (all_subtm f)),false)
  | And(f1,f2)   
  | Or(f1,f2)
  | Imply(f1,f2)
  | Eq(f1,f2)    -> 
    (TmSet.union (fst (all_subtm f1)) (fst (all_subtm f2)),false)
  | Not(f)       -> 
    (fst (all_subtm f), false)

(* input: a list of normalized patterns 
   works only with constant_type_flag := false *)

let subtm_hash = Hashtbl.create max_thms

let init_subtm_hash nh =
  let f _ tm = 
    let tmset = fst (all_subtm tm) in
    let g t =
      let n = try Hashtbl.find subtm_hash t with Not_found  -> 0. in
      Hashtbl.replace subtm_hash t (n +. 1.)
    in
    TmSet.iter g tmset
  in
  Hashtbl.clear subtm_hash;
  Hashtbl.iter f nh

let rec size_of_subtm t = match t with 
  | C(c,ty)      -> 1
  | App(f1,f2)   -> size_of_subtm f1 + size_of_subtm f2
  | _ -> failwith "size_of_subtm"

let mitigate_subtm_hash () =
  let mitigate k v = v /. float_of_int (size_of_subtm k) in
  hash_map mitigate subtm_hash  

let sort_subtm_hash () =
  let l = alist_of_hash subtm_hash in
  let compare_score (_,x) (_,y) = compare y x in
  List.sort compare_score l

let write_subtm l =
  let to_string (tm,score) = 
    (string_of_float score ^ ": " ^ string_of_subterm tm) 
  in
  mkdir "subterms";
  writel "subterms/subterms" (List.map to_string l)

let find_subtm () =
  init_subtm_hash norm_hash;
  mitigate_subtm_hash ();
  sort_subtm_hash ()
  
let find_frequent_subterm () =
  let l = find_subtm () in
  List.map fst (List.filter (fun (_,x) -> x >= 2.)  l)

(*  rlwrap ./top -I ../hh1 -I ..
open Toolbox;;
open Pattern;;
let palibs_dir = "../../../hh2-data/palibs"
let h4_dir = palibs_dir ^ "/" ^ "h4-kananaskis10/standard_library";;
let miz_dir = palibs_dir ^ "/" ^ "mizar/xml2_out";;
let mat_dir = palibs_dir ^ "/" ^ "matita/matita_out_new";;
let coq_dir = palibs_dir ^ "/" ^ "Coq-matching";;
let hl_dir = palibs_dir ^ "/" ^ "hl-225/standard_library";;
let (tyl,cl,thml,_) = Init.init_dir_prefix "h4/" h4_dir;;
set_prover 1;;
constant_counter := 0;
declare_constl tyl;
declare_constl cl;
init_norm_hash thml; 
write_subtml (find_subtm ());
*)

(*--------------------------------------------------------------------------
  Type inference for type application of constants
  -------------------------------------------------------------------------- *)

let rec strip_comb_ty_aux t = match t with
    App(f1,f2) -> let (tyoper,argl) = strip_comb_ty_aux f1 in 
                  (tyoper,argl @ [f2])
  | C(n,ty) -> (C(n,ty),[])
  | D(ty) -> (D(ty),[])
  | _ -> failwith "strip_comb_ty_aux"

let strip_comb_ty t = match strip_comb_ty_aux t with
  | (C(n,ty),l) -> (ty,l)
  | _ -> failwith "strip_comb_ty"

let rec type_subst v ty tm =
  if tm = v 
  then ty 
  else sub_tm_deep (type_subst v ty) tm 

let rec type_inference (ty,argl) = match (ty,argl) with
    (_,[]) -> ty
  | (Imply(x,y), _) -> type_inference (y, List.tl argl)
  | (Forall(v,tm), _) ->
      let argty = try List.hd argl with _ -> failwith "type_inference" in
      type_inference (type_subst v argty tm,  List.tl argl)
  | _ -> failwith "type_inference"

(*--------------------------------------------------------------------------
  Declare subterms
  -------------------------------------------------------------------------- *)
let subterm_hash1 = Hashtbl.create max_thms
let subterm_hash2 = Hashtbl.create max_thms
let subterm_hash () = match !prover_ref with
    1 -> subterm_hash1
  | 2 -> subterm_hash2
  | _ -> failwith "no prover_ref specified"

let declare_subterml subtml =
  let sh = subterm_hash () in
  Hashtbl.clear sh;
  let g tm = if !constant_type_flag 
             then try type_inference (strip_comb_ty tm) with _ -> 
                  begin 
                  let lref = inference_errorl () in
                  lref := tm :: !lref;
                  D("$true")
                  end
             else D("$true")
  in
  let l = List.map (fun x -> (string_of_subterm x, g x, x)) subtml in
  let f (name,ty,tm) = 
    Hashtbl.add sh name tm;
    declare_constant name ty 
  in
  List.iter f l

(*--------------------------------------------------------------------------
  Subtitute subterms by a constant
  -------------------------------------------------------------------------- *)

(* List all the possible substitutions  for one term *)
let create_substl frequent_subterml tm =
  let l = list_inter frequent_subterml (TmSet.elements (fst (all_subtm tm))) in
  let substl = 
    let f subtm = 
      let id = string_of_subterm subtm in 
      (subtm, C(both_of_constant (constant_hash ()) id))
    in
    List.map f l
  in
  substl

(* Apply substitution from top to bottom *)
let rec apply_substl substl tm = 
  if List.mem tm (List.map fst substl) 
  then List.assoc tm substl
  else sub_tm_deep (apply_substl substl) tm

let thm_variant_hash1 = Hashtbl.create max_thms
let thm_variant_hash2 = Hashtbl.create max_thms

let thm_variant_hash () = match !prover_ref with  
    1 -> thm_variant_hash1
  | 2 -> thm_variant_hash2
  | _ -> failwith "no prover_ref specified"


let create_thm_variant frequent_subterml (thm,tm) = 
  let tvh = thm_variant_hash () in
  let substl = create_substl frequent_subterml tm in
  if substl = [] 
  then None 
  else 
    (
    let name = add_prime (remove_prime thm ^ "__variant") in
    (* use seq_hash instead 
    Hashtbl.add time_thm_hash name (time_of_thm thm); *)
    Hashtbl.add tvh thm name;
    Some (name, (apply_substl substl tm))
    )

let create_thm_variants frequent_subterml =
  let tvh = thm_variant_hash () in 
  Hashtbl.clear tvh;
  let l = ref [] in
  let f thm tm = match create_thm_variant frequent_subterml (thm,tm) with
    None -> () 
  | Some x -> l := x :: !l
  in
  Hashtbl.iter f norm_hash;
  !l

(*--------------------------------------------------------------------------
  Abstraction
  -------------------------------------------------------------------------- *)

let var_dict = ref TmMap.empty		
let var_n = ref 0
let const_dict = ref TmMap.empty		
let const_n = ref 0
let pattern_dict = ref TmMap.empty
let pattern_n = ref 0

let rec abstract_var tm = 
match tm with
  | V(v,ty) -> V(find_int var_dict var_n tm, abstract_var ty)
  | _       -> sub_tm_deep abstract_var tm

let rec abstract_const tm = match tm with
  | C(c,ty) -> C(find_int const_dict const_n tm, abstract_const ty)
  | _       -> sub_tm_deep abstract_const tm 

let abstract tm = 
  reset_dict var_dict var_n;
  reset_dict const_dict const_n;
  abstract_const (abstract_var tm)

(*--------------------------------------------------------------------------
  Creating pattern dictionnaries
  -------------------------------------------------------------------------- *)

let tp_hash1 = Hashtbl.create max_thms
let tp_hash2 = Hashtbl.create max_thms

let tp_hash () = match !prover_ref with  
    1 -> tp_hash1
  | 2 -> tp_hash2
  | _ -> failwith "no prover_ref specified"

let tlp_hash = Hashtbl.create max_thms

let mk_pattern tm = 
  let numberc c = match c with C(n,ty) -> n | _ -> failwith "numberc" in     
  let cl = List.map fst (List.sort compare_snd (TmMap.bindings (!const_dict))) in
  let cl' = List.map numberc cl in
    (find_int pattern_dict pattern_n tm, cl')

(* normalization should already be done at that point *)
let update_thm_pattern tph ((thm:string),norm_tm) =
  let com_tm = if !cnfcom_flag 
               then fst (commute norm_tm)
               else norm_tm
  in
  let abs_tm = abstract com_tm in
  let (n,cl) = mk_pattern abs_tm in
  (* Transforming strings into int ref *) 
  (* Special case each constant is at the location of its index *)
  let arr = constant_array () in
  let f c = ref (Array.get arr c) in
  Hashtbl.replace tlp_hash thm (abs_tm,cl);
  Hashtbl.replace tph thm (n,List.map f cl) 

(* initialize theorems *)
let init_norm_hash thml =
  let f (thm,tm) =
    let tm = fmt_tm (format tm) in
    let simple_type_tm =
      if !simple_type_flag then simplify_type tm
      else tm
    in
    let norm_tm = 
      if !cnfcom_flag 
      then normalize simple_type_tm 
      else simple_type_tm
    in
    Hashtbl.add norm_hash thm norm_tm
  in
  Hashtbl.clear norm_hash;
  List.iter f thml

let nb_of_subterms = ref 0

(* simplify types before looking at subterms *)
let init_tp_hash (tyl,cl,thml) =
  let tph = tp_hash () in
  Hashtbl.clear tph;
  Hashtbl.clear tlp_hash;
  constant_counter := 0;
  norm_steps_warning := 0;
  (* declare and normalize objects *)
  declare_constl tyl;
  declare_constl cl;
  init_norm_hash thml;
  log_endline ("Warning: " ^ string_of_int (!norm_steps_warning) ^ 
               " deep normalization stopped.");
  (* option to add some frequent subterms *)  
  let norm_thml =
    if !subterm_flag
    then 
      let frequent_subterml = find_frequent_subterm () in
      nb_of_subterms := List.length (frequent_subterml);
      declare_subterml frequent_subterml;
      let new_thml = create_thm_variants frequent_subterml in
      alist_of_hash norm_hash @ new_thml
    else alist_of_hash norm_hash
  in
  (* save the total number of constants (not used in the new matching) *)
  let nbref = nb_of_constants () in 
  nbref := !constant_counter;
  (* number the patterns and create tp_hash *)
  List.iter (update_thm_pattern tph) norm_thml

(*--------------------------------------------------------------------------
  Pattern frequency (* not used anymore *)
  -------------------------------------------------------------------------- *)

let pattern_frequency1 = Hashtbl.create max_thms
let pattern_frequency2 = Hashtbl.create max_thms

let pattern_frequency () = match !prover_ref with  
    1 -> pattern_frequency1
  | 2 -> pattern_frequency2
  | _ -> failwith "no prover_ref specified"

let init_pattern_freq () =
  let tph = tp_hash () in
  let pf = pattern_frequency () in
  let f thm (n,cl) =
    try 
      let nbref = Hashtbl.find pf n in
      incr nbref;
    with _ ->
      Hashtbl.add pf n (ref 1)
  in     
    Hashtbl.clear pf; Hashtbl.iter f tph
  
(*--------------------------------------------------------------------------
  Remove single patterns in theorem-pattern hash. (to be adpated for 
  multiple provers) (to be removed)
  -------------------------------------------------------------------------- *)

let remove_lone_pattern () =
  let tph = tp_hash () in
  let h = Hashtbl.copy tph in
  let pf = pattern_frequency () in
  let f k (p,_) =
    if !(Hashtbl.find pf p) = 1 
    then Hashtbl.remove tph k 
    else ()
  in
  Hashtbl.iter f h

(*--------------------------------------------------------------------------
  Internal match
  -------------------------------------------------------------------------- *)

let internal_match = ref []
let mem_intern_cong1 = ref []
let mem_intern_cong2 = ref []

let mem_intern_cong () = match !prover_ref with  
    1 -> mem_intern_cong1
  | 2 -> mem_intern_cong2
  | _ -> failwith "no prover_ref specified"


let intern_concat sl =
  "'" ^ String.concat "//" (List.map remove_prime sl) ^ "'"

let class_of a m =
  let l = List.filter (fun x -> List.mem a x) m in
  match l with
  | [] -> [a]
  | [l'] -> l'
  | _    -> failwith "class_of"

let remainder_of a b m =
  List.filter (fun x -> not (List.mem a x) && not (List.mem b x)) m

let unite a b m =
  mk_set (class_of a m @ class_of b m) :: remainder_of a b m

let mem_cong mem_ref cg = (* assign the same number to all elements of a class *)
  let ch = constant_hash () in
  let ich = inv_constant_hash () in
  let namel = List.map (name_of_cint ich) cg in
  let common_name = intern_concat namel in
  let ty = try type_of_cint ich (List.hd cg) with _ -> failwith "mem_cong" in
  let n = !constant_counter in
  let f c = internal_match := (c,n) :: !internal_match in
  List.iter f cg;
  mem_ref := (namel,common_name) :: !mem_ref;
  Hashtbl.replace ch common_name (n,ty);
  Hashtbl.replace ich n (common_name,ty);
  incr constant_counter

let init_internal_match () =
  let congl = ref [] in
  let mem_ref = mem_intern_cong () in
  let sh = subterm_hash () in
  mem_ref := [];
  internal_match := [];
  let is_eq prop = match (fst prop) with 
    | Eq(C(n1,_),C(n2,_)) -> Some (n1,n2)
    | _  -> None
  in
  let itr _ prop = match is_eq prop with
    | Some (n1,n2) -> 
        begin
        let (c1,c2) = List.nth (snd prop) n1, List.nth (snd prop) n2 in
        let ich = inv_constant_hash () in
        let (c1s,c2s) = (name_of_cint ich c1, name_of_cint ich c2) in
        if !no_subterm_congruence_flag && (Hashtbl.mem sh c1s || Hashtbl.mem sh c2s)
        then ()
        else congl := unite c1 c2 !congl
        end
    | None -> ()
  in
    Hashtbl.iter itr tlp_hash;
    List.iter (mem_cong mem_ref) !congl

let apply_internal_match () =
  let arr = constant_array () in
  (* uses the fact that the adress of c is c *)
  let f (c,cmatch) = arr.(c) := cmatch in 
  List.iter f !internal_match


(*--------------------------------------------------------------------------
  Constant -> Property dictionnary 
  (somehow used to calculate the number of properties a constant have)
  Maybe not that important.
  -------------------------------------------------------------------------- *)

let const_prop_hash1 = Hashtbl.create max_constants
let const_prop_hash2 = Hashtbl.create max_constants

let const_prop_hash () = match !prover_ref with  
    1 -> const_prop_hash1
  | 2 -> const_prop_hash2
  | _ -> failwith "no prover_ref specified"

let prop_of_c c (n,cl) = 
  let f c1 c2 = if !(!c1) = !(!c2) then ref (ref this_cint) else c2 in		
  (n, List.map (f c) cl)

let update_const_prop cph p c =
  let value = try Hashtbl.find cph !(!c) with Not_found -> [] in
  let newvalue = prop_of_c c p :: value in 
  let key = !(!c) in
  Hashtbl.replace cph key newvalue
			
let update_const_propl cph _ ((_,cl) as p) = 
  List.iter (update_const_prop cph p) cl

let init_const_prop () =
  let tph = tp_hash () in
  let cph = const_prop_hash () in
  Hashtbl.clear cph;
  Hashtbl.iter (update_const_propl cph) tph

(*--------------------------------------------------------------------------
  Main function
  -------------------------------------------------------------------------- *)

let clear_match () =
  reset_dict pattern_dict pattern_n;
  init_array ();
  init_constant_hash ();
  init_errorl ()

let print_errorl () =
  let l1 = declaration_errorl () in
  let l2 = inference_errorl () in
  let n1 = List.length !l1 in
  let n2 = List.length !l2 in
  if n1 > 0 then
  log_endline ("Warning: " ^ string_of_int n1 ^ " constants not declared.")
  else ();
  if n2 > 0 then
  log_endline ("Warning: " ^ string_of_int n2 ^ 
    " type inference for subterms failed.")
  else ()

(* Use clear_match before using this function on a prover 
   but do not clear before using it on the second prover of a pair *)
let init_dicts (tyl,cl,thml) =
  init_tp_hash (tyl,cl,thml);
  init_pattern_freq ();
  if !no_congruence_flag 
  then ()
  else 
    begin
    init_internal_match ();
    apply_internal_match ()
    end;
  init_const_prop ();
  print_errorl ();
  Hashtbl.clear tlp_hash

let init_dicts_noclear (tyl,cl,thml) =
  init_tp_hash (tyl,cl,thml);
  init_pattern_freq ();
  if !no_congruence_flag 
  then ()
  else 
    begin
    init_internal_match ();
    apply_internal_match ()
    end;
  init_const_prop ();
  print_errorl ();
  ()





