open Hh_term
open Toolbox
open Init
open Pattern

let inv_epsilon = 1000
let epsilon = 1. /. float_of_int (1000)
(* score associated with each property representant *)
let p_hash  = Hashtbl.create max_thms
let cc_hash = Hashtbl.create max_thms
let tt_hash  = Hashtbl.create max_thms
let ccscore_hash = Hashtbl.create max_thms
let ttscore_hash = Hashtbl.create max_thms
let cccoeff_hash = Hashtbl.create max_thms
let ttcoeff_hash = Hashtbl.create max_thms
let thml1_init_glob = ref []
let thml2_init_glob = ref []

(*--------------------------------------------------------------------------
  Identifies some logical constants (essential for mizar mappings)
  -------------------------------------------------------------------------- *)

let rec replace_tyimp_aux t = match t with
  | Id ">" -> Id "=>"
  | Id "$t" -> Id "$true" 
  | Id "$tType" -> Id "$true"
  | Comb(x,y) -> Comb(replace_tyimp_aux x, replace_tyimp_aux y)
  | Abs(v,ty,tm) -> Abs(v, replace_tyimp_aux ty, replace_tyimp_aux tm)
  | _ -> t 

let replace_tyimp (s,t) = (s,replace_tyimp_aux t)

let replace_tyimp_allobj (a,b,c) = 
  (List.map replace_tyimp a, List.map replace_tyimp b, List.map replace_tyimp c)

(*--------------------------------------------------------------------------
  Global reference for slow motion
  -------------------------------------------------------------------------- *)

let slow_mo_flag = ref false
let slow_mo_mem = ref []
let slow_mo_max = ref 0
let auto_max = ref 500
let auto_counter = ref 0
let self_flag = ref false

let set_no_type () =
  (constant_type_flag := false; 
   simple_type_flag := false; 
   no_type_flag := true)

let set_simple_type () =
  (constant_type_flag := false; 
   simple_type_flag := true;
   no_type_flag := false)

let set_var_type () =
  (constant_type_flag := false; 
   simple_type_flag := false;
   no_type_flag := false)

let set_constant_type () =
  (constant_type_flag := true; 
   simple_type_flag := false;
   no_type_flag := false)

(*--------------------------------------------------------------------------
  Create a p_hash for 2 provers
  -------------------------------------------------------------------------- *)

let deref cl = List.map (fun x -> !(!x)) cl

let init_two_dicts objects1 objects2 =
  clear_match (); (* important *)
  set_prover 1; time "init1" init_dicts objects1;
  set_prover 2; time "init2" init_dicts objects2

(* used for statistics *)
let init_p_hash_onelib () =
  Hashtbl.clear p_hash;
  let f1 thm (p,cl) = 
    if cl = [] then ()
    else
      let l = try Hashtbl.find p_hash p with Not_found -> [] in
      Hashtbl.replace p_hash p ((1,thm,deref cl) :: l)
  in
  Hashtbl.iter f1 tp_hash1

let init_p_hash objects1 objects2 =
  Hashtbl.clear p_hash;
  time "norm" (init_two_dicts objects1) objects2;
  let f1 thm (p,cl) = 
    if cl = [] then ()
    else
      let l = try Hashtbl.find p_hash p with Not_found -> [] in
      Hashtbl.replace p_hash p ((1,thm,deref cl) :: l)
  in
  let f2 thm (p,cl) = 
    if cl = [] then () 
    else
      let l = try Hashtbl.find p_hash p with Not_found -> [] in
      Hashtbl.replace p_hash p ((2,thm,deref cl) :: l)
  in  
  Hashtbl.iter f1 tp_hash1;
  Hashtbl.iter f2 tp_hash2

(*--------------------------------------------------------------------------
  Building hash
  -------------------------------------------------------------------------- *)

let append_cc_hash h cc v =
  let l = try Hashtbl.find h cc with Not_found -> 
    let n1 = List.length (Hashtbl.find const_prop_hash1 (fst cc)) in
    let n2 = List.length (Hashtbl.find const_prop_hash2 (snd cc)) in
    let coeff = 1. /. log (2. +. float_of_int (n1 * n2)) in
    Hashtbl.replace cccoeff_hash cc (coeff,1.,coeff);
    []
  in
  Hashtbl.replace h cc (v :: l)

let cross_matching l =
  let (l1,l2) = List.partition (fun (a,_,_) -> a = 1) l in
  let l1' = List.map (fun (_,thm,cl) -> (thm,cl)) l1 in
  let l2' = List.map (fun (_,thm,cl) -> (thm,cl)) l2 in
  let n1 = List.length l1' in
  let n2 = List.length l2' in
  if n1 * n2 > inv_epsilon then []
  else
  let rl = cartesian_product l1' l2' in
  let f ((t1,cl1),(t2,cl2)) = ((t1,t2), List.combine cl1 cl2) in
  List.map f rl

let init_cctt_hash ph =
  Hashtbl.clear cc_hash;
  Hashtbl.clear tt_hash;
  Hashtbl.clear ccscore_hash;
  Hashtbl.clear ttscore_hash;
  Hashtbl.clear cccoeff_hash;
  Hashtbl.clear ttcoeff_hash;
  let add_entry pn coeffs ((thm1,thm2),ccl) =
    Hashtbl.replace ttcoeff_hash (thm1,thm2) coeffs;
    Hashtbl.add tt_hash (thm1,thm2) (pn,ccl);
    Hashtbl.add ttscore_hash (thm1,thm2) 0.;
    let f (thm1,thm2) cc = 
      append_cc_hash cc_hash cc (thm1,thm2);
      Hashtbl.replace ccscore_hash cc 1.  
    in
    List.iter (f (thm1,thm2)) ccl
  in
  let iter pn l =
    if l = [] then () else begin 
    let (_,_,cl) = List.hd l in
    let entryl = 
      if !self_flag 
      then List.filter (fun ((a,b),_) -> a <> b) (cross_matching l)
      else cross_matching l
    in
    if entryl = [] then () else begin
    let f1 = 1. /. log (2. +. float_of_int (List.length cl)) in
    let f2 = 1. /. log (2. +. float_of_int (List.length entryl)) in
    let coeffs = (f1,f2,f1 *. f2) in
    List.iter (add_entry pn coeffs) entryl
    end
    end
  in
  Hashtbl.iter iter ph

(*--------------------------------------------------------------------------
  Tests
  -------------------------------------------------------------------------- *)

let is_neg cc = Hashtbl.find ccscore_hash cc < 0.
let is_pos cc = Hashtbl.find ccscore_hash cc > 1.
let is_undec cc = 
  let x = Hashtbl.find ccscore_hash cc in 
  x >= 0. && x <= 1.  
let is_undec_ccscore (cc,x) = x >= 0. && x <= 1.  
let is_good_ccscore (cc,x) = x >= 0.3 && x <= 1. 

let is_dec_ccscore (cc,x) = not (x >= 0. && x <= 1.)
let is_neg_ccscore (cc,x) = x < 0.
let is_pos_ccscore (cc,x) = x > 1.
let is_dec cc = not (is_undec cc)

(*--------------------------------------------------------------------------
  Printing
  -------------------------------------------------------------------------- *)

(* Printing theorems *)
let ttscore_to_string ((t1,t2),f) = (string_of_float f ^ " " ^ t1 ^ " " ^ t2)

(* Printing constants *)
let cc_to_string (c1,c2) = 
  (name_of_cint inv_constant_hash1 c1) ^ " || " ^ 
  (name_of_cint inv_constant_hash2 c2)

let name_of_cc (c1,c2) = 
  ((name_of_cint inv_constant_hash1 c1), (name_of_cint inv_constant_hash2 c2))
  
let pretty_cc (c1,c2) =
  (name_of_cint inv_constant_hash1 c1) ^ " || " ^ 
  (name_of_cint inv_constant_hash2 c2)

let pretty_ccscore (cc,f) =
  pretty_cc cc ^ " || " ^ string_of_float f

let pretty_ccscore' ((c1,c2),f) =
  let n = String.length ((name_of_cint inv_constant_hash1 c1)) in
  (String.make n ' ' ^ " || " ^ (name_of_cint inv_constant_hash2 c2)
   ^ " || " ^ string_of_float f)

let ccscore_to_name ((c1,c2),score) =
  ((name_of_cint inv_constant_hash1 c1, 
    name_of_cint inv_constant_hash2 c2),score)

(* Printing graphs *)
let stringl_of_scorel scorel =
  let i  = ref 0 in
  let l0 = List.map (fun x -> incr i; (!i,x)) scorel in
  let trunc x = try Str.first_chars x 5 with _ -> x in
  let to_string (a,b) = 
    string_of_int a ^ " " ^ trunc (string_of_float b)
  in
  "position score" :: (List.map to_string l0)

let graph_of n xxscorel =
  stringl_of_scorel (List.map snd (first_n n xxscorel))

(* Printing results *)
let write_match_result expname resultl = 
  let file_out = expname ^ "/interactive_match" in
  let os name value = append file_out (name ^ ": " ^ value ^ "\n") in
  let ccscorel = alist_of_hash ccscore_hash in
  let ttscorel = alist_of_hash ttscore_hash in
  log_endline "  Properties"; 
  (* Properties *)
  let propl = List.map snd (alist_of_hash p_hash) in
  let is_in_both l = 
    let (l1,l2) = List.partition (fun (x,_,_) -> x = 1) l in
    l1 <> [] && l2 <> []
  in
  let common_propl = List.filter is_in_both propl in
  let props = string_of_int (List.length propl) in
  let common_props = string_of_int (List.length common_propl) in
  os "Property" props;
  os "Common property" common_props;
  (* Internal matches *)
  writel (expname ^ "/internal1") (List.map snd !mem_intern_cong1);
  writel (expname ^ "/internal2") (List.map snd !mem_intern_cong2);
  log_endline "  Internal matches";
  (* Decided matches *)
  let resultl_rev = List.rev resultl in
  let cc_decided = string_of_int (List.length resultl) in
  os "CC positive" cc_decided;
  writel (expname ^ "/cc_decided") (List.map pretty_ccscore resultl_rev);
  writel (expname ^ "/cc_decided_graph") (graph_of 1000 resultl_rev);
  log_endline "  Decided matches"; 
  (* Undecided matches *)
  let undecl = List.filter is_undec_ccscore ccscorel in
  let undecl = List.sort compare_score undecl in
  let undecl = 
    if !self_flag 
    then List.filter (fun ((c1,c2),score) -> c1 <> c2) undecl
    else undecl
  in
  let cc_undecided = string_of_int (List.length undecl) in
  os "CC undecided" cc_undecided;
  writel (expname ^ "/cc_undecided") (List.map pretty_ccscore undecl);
  writel (expname ^ "/cc_undecided_graph") (graph_of 1000 undecl);
  let n_undecided = string_of_int (List.length undecl) in
  log_endline ("  " ^ n_undecided ^ " undecided matches");
  (* Negative matches *)
  let negl =
    List.sort compare_score (List.filter is_neg_ccscore ccscorel)
  in
  let cc_neg = string_of_int (List.length resultl) in
  os "CC negative" cc_neg;
  writel (expname ^ "/cc_neg") (List.map pretty_ccscore negl);
  writel (expname ^ "/cc_neg_graph") (graph_of 1000 negl);
  log_endline "  Negative matches"; 
  (* Theorems matches *)
  if !self_flag then () else
  begin
    let ttsortedl = List.sort compare_score ttscorel in
    let tt_length = string_of_int (List.length ttsortedl) in
    os "TT" tt_length;
    writel (expname ^ "/tt") (List.map ttscore_to_string ttsortedl);
    writel (expname ^ "/tt_graph") (graph_of 1000 ttsortedl);
    log_endline "  Theorem matches"
  end

(*--------------------------------------------------------------------------
  Views
  -------------------------------------------------------------------------- *)

let view_left l =
  log_endline "";
  let i = ref 1 in
  let c = ref (-1) in 
  let iter ((c1,c2),score) =
    let sn = string_of_int !i ^ " " in
    let s = if !c = c1 
            then pretty_ccscore' ((c1,c2),score)
            else (c := c1; pretty_ccscore ((c1,c2),score))
    in
    log_endline (sn ^ s);
    incr i
  in
  List.iter iter l;
  log_endline ""

let view_plain l = 
  log_endline "";
  let i = ref 1 in
  let iter ((c1,c2),score) =
    let sn = string_of_int !i ^ " " in
    let s = pretty_ccscore ((c1,c2),score) in
    log_endline (sn ^ s);
    incr i
  in
  List.iter iter l;
  log_endline ""

(*--------------------------------------------------------------------------
  Updating the score_hash by giving advice
  -------------------------------------------------------------------------- *)

(* global reference for interactive matching *)
let resultl_glob = ref []
let greedy_flag = ref false
let confidence_glob = ref 3.
let human_type_flag = ref false (* only when comparing a prover to hol4 or hollight *)
let type_coherence_flag = ref false
let disambiguate_flag = ref false
let sum_to_prod_flag = ref false

(* test *)
let app_ccscore cc = (cc, Hashtbl.find ccscore_hash cc)

(* after normalization *)
let rec is_a_type ty = match ty with
    D("$true") -> true
  | Imply(D("$true"),b) -> is_a_type b
  | _ -> false

let is_type_match ((c1,c2),_) =
  is_a_type (type_of_cint inv_constant_hash1 c1) ||
  is_a_type (type_of_cint inv_constant_hash2 c2)
  
(* advice *)
let neg_advice (cc,score) =
  Hashtbl.replace ccscore_hash cc (-1.)

let greedy_advice (c1,c2) =
  let f (c1',c2') _ =
    if (c1 = c1' || c2 = c2') && (c1',c2') <> (c1,c2) 
    then neg_advice ((c1',c2'),0.)
    else ()
  in
  Hashtbl.iter f ccscore_hash

let pos_advice_list = ref []

let is_negative cc = 
  (try Hashtbl.find ccscore_hash cc with _ -> 1.) < 0.
  
let is_denied (c1,c2) = 
  let l = List.map fst !resultl_glob in
  let l1 = List.map fst l in
  let l2 = List.map snd l in
  (List.mem c1 l1 || List.mem c2 l2) && (not (List.mem (c1,c2) l))

let match_types (c1,c2) =  
  let ty1 = type_of_cint inv_constant_hash1 c1 in
  let ty2 = type_of_cint inv_constant_hash2 c2 in
  let cl1 = cint_of_hhm ty1 in
  let cl2 = cint_of_hhm ty2 in
  let tyl = List.combine cl1 cl2 in
  if List.exists is_negative tyl then failwith "match_types";
  if !greedy_flag && List.exists is_denied tyl then failwith "match_types";
  let tyl1 = List.map (fun x -> x,0.) tyl in
  let tyl2 = List.filter (fun (cc,s) -> (try is_undec cc with _ -> true)) tyl1 in
    pos_advice_list := !pos_advice_list @ tyl2

let rec pos_advice_loop (cc,score) = 
  pos_advice_list := (cc,score) :: !pos_advice_list;
  if !type_coherence_flag 
  then try match_types cc with _ -> 
        (
         log_endline ("Denied: " ^ pretty_ccscore (cc,score));
         neg_advice(cc,score);
         pos_advice_list := [];
        )
  else ()


let pos_save (cc,score) =
  Hashtbl.replace ccscore_hash cc !confidence_glob;
  resultl_glob := (cc,score) :: !resultl_glob;
  if !greedy_flag then greedy_advice cc else ()

let pos_advice (cc,score) =
  pos_advice_list := [];
  pos_advice_loop (cc,score);
  if !human_type_flag && List.exists is_type_match !pos_advice_list
  then 
    begin  
    let l = List.filter is_type_match !pos_advice_list in
    if List.mem cc (List.map fst l) 
      then view_plain l
      else view_plain ((cc,score) :: l);
    let s = read_line () in
      let rec human_loop s1 =   
      match s1 with
        "y" -> List.iter pos_save !pos_advice_list
      | "n" -> neg_advice (cc,score)
      | "break" -> auto_max := 0
      | _   -> (log_endline "Invalid_input"; 
                let s2 = read_line () in human_loop s2)
    in
    human_loop s
    end
  else List.iter pos_save !pos_advice_list
   
let reset_advice (cc,score) =
  Hashtbl.replace ccscore_hash cc (1.);
  resultl_glob := List.filter (fun (cc',_) -> cc' <> cc) !resultl_glob

(*--------------------------------------------------------------------------
  Scoring functions
  -------------------------------------------------------------------------- *)

let rec product l = 
  if l = [] then 1. else (List.hd l) *. product (List.tl l)

let mylog x = log (2. +. x)

let to_proba x = x /. (x +. 1.) 

let has_changed = ref false (* to test the convergence *)

let disambiguate_hash1 = Hashtbl.create max_constants
let disambiguate_hash2 = Hashtbl.create max_constants

(* Disambiguation *)
let ambiguity1 undecl =
  let f ((c1,_),score) = 
    let old_sum = try Hashtbl.find disambiguate_hash1 c1 with _ -> 0. in
    Hashtbl.replace disambiguate_hash1 c1 (old_sum +. score)
  in
  List.iter f undecl

let ambiguity2 undecl =
  let f ((_,c2),score) = 
    let old_sum = try Hashtbl.find disambiguate_hash2 c2 with _ -> 0. in
    Hashtbl.replace disambiguate_hash2 c2 (old_sum +. score)
  in
  List.iter f undecl

let disambiguate_hash undecl =
  Hashtbl.clear disambiguate_hash1;
  Hashtbl.clear disambiguate_hash2;
  ambiguity1 undecl;
  ambiguity2 undecl;
  let n = List.length undecl in
  let h = Hashtbl.create n in
  let ll x = 1. /. (log (10. +. x)) in
  let f ((c1,c2),_) = 
    let f1 = Hashtbl.find disambiguate_hash1 c1 in
    let f2 = Hashtbl.find disambiguate_hash2 c2 in
    (* maintain this coefficent between 0 and 1 to keep scores between 0 and 1 *)
    let cf = ll ((1. +. f1) *. (1. +. f2)) in 
    Hashtbl.add h (c1,c2) cf
  in
  List.iter f undecl;
  h

(* Main scores *)
let update_score_tt () =
  let iter tt (pn,ccl) =
    let (_,_,coeff) = Hashtbl.find ttcoeff_hash tt in    
    let l = List.map (Hashtbl.find ccscore_hash) ccl in
    let score = 
      if List.exists (fun x -> x < 0.) l then 0.
      else coeff *. sum l 
    in
    Hashtbl.replace ttscore_hash tt score
  in
  Hashtbl.iter iter tt_hash

let update_score_cc () =
  let iter cc ttl =
    let old_score = Hashtbl.find ccscore_hash cc in
    if old_score < 0. || old_score > 1. then ()
    else
      begin
      let l = List.map (Hashtbl.find ttscore_hash) ttl in
      let (_,_,coeff) = Hashtbl.find cccoeff_hash cc in
      let score = to_proba (coeff *. sum l) in
      if abs_float (score -. old_score) > epsilon
      then has_changed := true else ();
      Hashtbl.replace ccscore_hash cc score
      end
  in
  Hashtbl.iter iter cc_hash

(*--------------------------------------------------------------------------
  Constant dependency coefficients or a way to update your function in one go.
  -------------------------------------------------------------------------- *)

let rec regroup_aux h l = match l with
    [] -> ()
  | (cc,coeff) :: m -> 
    begin
      let prev_coeff = try Hashtbl.find h cc with _ -> 0. in
      Hashtbl.replace h cc (prev_coeff +. coeff);
      regroup_aux h m
    end
 
let regroup_sum l = 
  let h = Hashtbl.create (List.length l) in
  regroup_aux h l;
  h


let dependency_coeff_hash = Hashtbl.create max_constants
let ttscorel_hash = Hashtbl.create max_thms
 
let dependency_coeff_tt () =
  let iter tt (pn,ccl) =
    let (_,_,coeff) = Hashtbl.find ttcoeff_hash tt in    
    let scorel = List.map (fun cc -> (cc,coeff)) ccl in
    Hashtbl.replace ttscorel_hash tt scorel
  in
  Hashtbl.iter iter tt_hash


let dependency_coeff_cc () =
  let iter cc ttl =
    let ll = List.map (Hashtbl.find ttscorel_hash) ttl in
    let h = regroup_sum (List.concat ll) in
    let (_,_,coeff) = Hashtbl.find cccoeff_hash cc in
    hash_map (fun cc' coeff' -> coeff *. coeff') h;
    Hashtbl.replace dependency_coeff_hash cc h
  in
  Hashtbl.iter iter cc_hash

let init_dependency_coeff_hash () =
  Hashtbl.clear ttscorel_hash;
  Hashtbl.clear dependency_coeff_hash;
  dependency_coeff_tt ();
  dependency_coeff_cc ()

(* This gives the influence of the first pair over the second pair *)
let find_dep_coeff_error h (cc1,cc2) =
  let h' = Hashtbl.find h cc2 in Hashtbl.find h' cc1

let find_dep_coeff h (cc1,cc2) =
  try find_dep_coeff_error h (cc1,cc2) with _ -> 0.


(*--------------------------------------------------------------------------
  Loop
  -------------------------------------------------------------------------- *)

let match_init expname objects1 objects2 =
  log_endline "Initialization";
  init_p_hash objects1 objects2;
  time_file (expname ^ "/interactive_match") "init cc and tt hash: " 
    init_cctt_hash p_hash

let loop_counter = ref 0

let rec match_multiple_loop_aux () =
  if !slow_mo_flag 
  then slow_mo_mem := alist_of_hash ccscore_hash :: !slow_mo_mem
  else ();
  if !slow_mo_flag && !loop_counter >= !slow_mo_max then ()
  else begin
    incr loop_counter;
    has_changed := false;
    update_score_tt ();
    update_score_cc ();
    if !has_changed then match_multiple_loop_aux () else ()
  end

let zero_is_negative () = 
  hash_map (fun k v -> if v < epsilon then (-1.) else v) ccscore_hash
  
let match_multiple_loop () =
  loop_counter := 0;
  match_multiple_loop_aux ();
  zero_is_negative () (* made default recently *)

(*--------------------------------------------------------------------------
  Ordering (input is already sorted by scores)
  -------------------------------------------------------------------------- *)
  
let regroup_top_left l =
  if l = [] then ([],[]) else 
    let ((c1,c2),score) = List.hd l in
    let (l1,l2) = List.partition (fun ((c,_),_) -> c = c1) (List.tl l) in
    (List.hd l :: l1, l2)

let rec order_factor f l =
  if l = [] then []
  else let (l1, l2) = f l in l1 @ order_factor f l2

let order_left l = order_factor regroup_top_left l

let order_score l = l

let rec common_suffix_aux n1 n2 s1 s2 n =
  if n > n1 || n > n2 then (n - 1)
  else 
    if s1.[n1 - n] = s2.[n2 - n]
    then common_suffix_aux n1 n2 s1 s2 (n + 1)
    else (n - 1)

let common_suffix s1 s2 = 
  common_suffix_aux (String.length s1) (String.length s2) s1 s2 1

let order_string l = 
  let compare_sscore (_,(n1,_)) (_,(n2,_)) = compare n2 n1 in
  let f1 ((c1,c2),score) =
    let (c1s,c2s) = name_of_cc (c1,c2) in
    ((c1,c2),(common_suffix c1s c2s,score))
  in
  let f2 (cc,(_,score)) = (cc,score) in
  List.map f2 (List.sort compare_sscore (List.map f1 l))

let order_dec _ =
  let l = alist_of_hash ccscore_hash in
  List.filter (is_pos_ccscore) l @ List.filter (is_neg_ccscore) l
  
(*--------------------------------------------------------------------------
  Advice
  -------------------------------------------------------------------------- *)

(* global reference for interactive matching *)
let nbmatch_glob = ref 40
let order_glob = ref order_score
let view_glob = ref view_plain 

type continue_type = Update | Break | Refresh 

let result_glob_mem = ref []

(* automatic *)  
let auto_timer () = 
  incr auto_counter; 
  if !auto_counter >= !auto_max 
    then Break 
    else begin
         let l = list_diff !resultl_glob !result_glob_mem in
         result_glob_mem := !resultl_glob;
         log_endline ((string_of_int !auto_counter) ^ ": " ^ 
         String.concat " " (List.map pretty_ccscore l));
         Update
         end
(* don't worry about that it is the same thing *)


let break_advice l = Break

let first_advice l =
  if l = [] 
  then Break 
  else (pos_advice (List.hd l); auto_timer ())

(* interactive *)
type human_type = 
    Refresh_h 
  | Break_h 
  | Advice_h of (int list * int list) 
  | Reset_h of (int list) 

let rec mk_int_list a b =
  if a > b then [] else a :: mk_int_list (a + 1) b

let dest_advice_input input =
  let sl = Str.split (Str.regexp " +") input in
  let read_token token =
    match Str.split (Str.regexp ",") token with
      [a]   -> (try [int_of_string a] with _ -> failwith "dest_advice")
    | [a;b] -> (try
                let (a',b') = (int_of_string a, int_of_string b) in
                if compare a' b' < 0
                then mk_int_list a' b'
                else mk_int_list b' a'
                with _ -> failwith "dest_advice")  
    | _     -> failwith "dest_advice"
  in
  let l = List.concat (List.map read_token sl) in
  let (nl,pl) = List.partition (fun x -> x < 0) l in
  (List.map abs nl, pl)

let dest_reset input =
  let sl = Str.split (Str.regexp " +") input in
  match sl with
  | "reset" :: m -> List.map int_of_string m
  | _ -> failwith "dest_reset"

let is_advice_input s = can dest_advice_input s

let is_coherent_input (nl,pl) l =
  let f x = x > 0 && x < 1 + (List.length l) in
  List.for_all f nl && List.for_all f pl

let human_input l = 
  !view_glob l;
  let line = read_line () in
  let rec human_loop input = match input with
      "stop" -> Break_h
    | "undo" -> (order_glob := order_dec; Refresh_h)
    | "more" -> (nbmatch_glob := !nbmatch_glob + 10; Refresh_h)
    | "less" -> (nbmatch_glob := !nbmatch_glob - 10; Refresh_h)
    | "order left" -> 
      (view_glob := view_left; order_glob := order_left; Refresh_h)
    | "order string" ->  
      (view_glob := view_plain; order_glob := order_string; Refresh_h)
    | "order score" -> 
      (view_glob := view_plain; order_glob := order_score; Refresh_h)
    | _  when is_advice_input input -> 
      let (nl,pl) = dest_advice_input input in
      if is_coherent_input (nl,pl) l 
        then Advice_h (nl,pl) 
        else error "Input not coherent"
    | _ when can dest_reset input ->
      let nl = dest_reset input in
      if List.for_all (fun x -> x > 0 && x < 1 + (List.length l)) nl
      then Reset_h nl
      else error "Input not coherent"
    | _ -> error "Input not valid"
  and error msg = 
    log_endline msg; 
    let new_input = read_line () in human_loop new_input
  in
  human_loop line

let human_advice l = match human_input l with (* effects *)
    Break_h   -> Break
  | Refresh_h -> Refresh
  | Advice_h (nl,pl) -> 
    begin 
    List.iter (fun x -> neg_advice (List.nth l (x - 1))) nl;
    List.iter (fun x -> pos_advice (List.nth l (x - 1))) pl;
    Update
    end
  | Reset_h nl ->
    begin
    List.iter (fun x -> reset_advice (List.nth l (x - 1))) nl;
    Update
    end

(*--------------------------------------------------------------------------
  Main function: matching
  -------------------------------------------------------------------------- *)

let advice_glob = ref human_advice

let interactive_match expname objects1 objects2 =
  let file_out = expname ^ "/interactive_match" in
  mkdir_rec expname;
  erase_file file_out;
  auto_counter := 0;
  resultl_glob := [];
  time_file file_out "match_init: " 
    (match_init expname objects1) objects2;
  log_endline "First matching loop";
  time_file file_out "Scoring time (first iteration): " match_multiple_loop ();
  append file_out ("Number of loops (first iteration): " ^ 
                   string_of_int (!loop_counter) ^ "\n");
  let sort_undecided () = 
    let undecl = List.filter is_undec_ccscore (alist_of_hash ccscore_hash) in
    let new_undecl = 
      if !disambiguate_flag 
      then 
        let h = disambiguate_hash undecl in
        let f (cc,score) = (cc, Hashtbl.find h cc *. score) in
        List.map f undecl
      else undecl
    in
    List.sort compare_score new_undecl
  in
  let rec interactive_loop () =
    let bestl' = 
      if !self_flag
      then List.filter (fun ((c1,c2),score) -> c1 <> c2) (sort_undecided ())
      else sort_undecided ()
    in
    let bestl = !order_glob (first_n !nbmatch_glob bestl') in
    match !advice_glob bestl with (* effects *)
      Break   -> () 
    | Update  -> (match_multiple_loop (); interactive_loop ())
    | Refresh -> interactive_loop ()
  in
  interactive_loop ();
  log_endline "Writing results";
  write_match_result expname !resultl_glob

let match_provers_dir expname (dir1,dir2) =
  let ((ty1,cl1,thml1),_) = init_dir dir1 in
  let ((ty2,cl2,thml2),_) = init_dir dir2 in
  thml1_init_glob := thml1;
  thml2_init_glob := thml2;
  interactive_match expname 
    (replace_tyimp_allobj (ty1,cl1,thml1))
    (replace_tyimp_allobj (ty2,cl2,thml2))

let match_provers expname (prover1,prover2) =
  let ((ty1,cl1,thml1),_) = init_dir (lib_of prover1) in
  let ((ty2,cl2,thml2),_) = init_dir (lib_of prover2) in
  thml1_init_glob := thml1;
  thml2_init_glob := thml2;
  interactive_match expname 
    (replace_tyimp_allobj (ty1,cl1,thml1)) 
    (replace_tyimp_allobj (ty2,cl2,thml2))

(*--------------------------------------------------------------------------
  Transitive matches
  -------------------------------------------------------------------------- *)

let find_transitive l12 l23 l13 =
  let h_old = hash_of_alist l13 in
  let h = Hashtbl.create 100000 in
  let f (((a,b),score1), ((b',c),score2)) = 
    if b <> b' then ()
    else 
      let newscore = score1 *. score2 in
      let oldscore = try fst (Hashtbl.find h (a,c)) with _ -> -1. in
      if newscore > oldscore 
        then Hashtbl.replace h (a,c) (newscore,b)
        else ()
  in
  List.iter f (cartesian_product l12 l23);
  let add_old k (new_score,b) =
    let old_score = try (Hashtbl.find h_old k) with _ -> 0. in
    (new_score,old_score,b)
  in
  let res_h = hash_map_new add_old h in 
  let add_old_entry k v = 
    if Hashtbl.mem res_h k then () else Hashtbl.add res_h k (0.,v,"None")
  in
  Hashtbl.iter add_old_entry h_old;
  let res_l = alist_of_hash res_h in
  let compare_scor (_,(a,_,_)) (_,(a',_,_)) = compare a' a in
  let compare_sum (_,(a,b,_)) (_,(a',b',_)) = compare (a'+. b') (a +. b) in
  let compare_diff (_,(a,b,_)) (_,(a',b',_)) = compare (a'-. b') (a -. b) in
  let compare_opt (_,(a,b,_)) (_,(a',b',_)) = 
    compare ((b'+. 0.001) *. (a' -. b')) ((b +.0.001) *. (a -. b))
  in

  let scorel = List.sort compare_scor res_l in
  let suml = List.sort compare_sum res_l in
  let diffl = List.sort compare_diff res_l in
  let optl = List.sort compare_opt res_l in
  (scorel,suml,diffl,optl)









