(*-------------------------------------------------------------------------- *)
(* Useful functions used troughout the developments                          *)
(*-------------------------------------------------------------------------- *)

(*--------------------------------------------------------------------------
  Basic operators
  -------------------------------------------------------------------------- *)

let (++) f g = (fun x -> f (g x))
let can f s = try (f s; true) with _ -> false

let couple_of_list l = 
  if List.length l = 2
  then (List.nth l 0, List.nth l 1)
  else failwith "couple_of_list"

let triple_of_list l = 
  if List.length l = 3
  then (List.nth l 0, List.nth l 1, List.nth l 2)
  else failwith "triple_of_list"

let unsafe_triple_of_list l = (List.nth l 0, List.nth l 1, List.nth l 2)

let associate f x = (x, f x)

(*--------------------------------------------------------------------------
  String
  -------------------------------------------------------------------------- *)

let explode s =
  let rec expl i l =
    if i < 0 then l else
    expl (i - 1) (s.[i] :: l) in
  expl (String.length s - 1) []

let implode l =
  let result = String.create (List.length l) in
  let rec imp i = function
  | [] -> result
  | c :: l -> result.[i] <- c; imp (i + 1) l in
  imp 0 l

let contains s1 s2 =
    let re = Str.regexp_string s2
    in
        try ignore (Str.search_forward re s1 0); true
        with Not_found -> false

(*--------------------------------------------------------------------------
  Debugging
  -------------------------------------------------------------------------- *)

let append file str =
  let oc = open_out_gen [Open_creat; Open_text; Open_append] 0o640 file in
  output_string oc str;
  close_out oc

let log_file_glob = ref "log"
let log_string s = append !log_file_glob s
let log_only_endline s = log_string (s ^ "\n")

let no_print_log_flag = ref false

let log_endline s = 
  if !no_print_log_flag then () else print_endline s; 
  log_string (s ^ "\n")

let time name f x =
    let t = Sys.time() in
    let fx = f x in
    log_only_endline (name ^ ": " ^ string_of_float (Sys.time() -. t) ^ "sec");
    fx

let time_here name f x =
    let t = Sys.time() in
    let fx = f x in
    log_endline (name ^ ": " ^ string_of_float (Sys.time() -. t) ^ "sec");
    fx

(*--------------------------------------------------------------------------
  IO
  -------------------------------------------------------------------------- *)

let erase_file file =
  let oc = open_out file in 
    output_string oc "";
  close_out oc

let readl file = 
let lines = ref [] in
let chan = open_in file in
try
  while true; do
    lines := input_line chan :: !lines
  done; !lines
with End_of_file ->
  close_in chan;
  List.rev !lines ;;

let writel file strl =
  let oc = open_out file in 
    List.iter (Printf.fprintf oc "%s\n") strl; 
  close_out oc

let mkdir s = 
  try Unix.mkdir s 0o777 with Unix.Unix_error (Unix.EEXIST,_,_) -> ()

let rec mkdir_rec s = 
  if s = "." || s = ".." || s = "" || s = "/" then ()
  else (mkdir_rec (Filename.dirname s); mkdir s)

let time_file file name f x =
  let t = Sys.time() in
  let fx = f x in
  append file (name ^ ": " ^ string_of_float (Sys.time() -. t) ^ "sec\n");
  fx

let clean_dir dir =
  (try ignore (Sys.command ("rm -r " ^ dir)) with _ -> ());
  mkdir_rec dir

(*--------------------------------------------------------------------------
  List
  -------------------------------------------------------------------------- *)

let rec first_n n l =
  if n <= 0 || l = []
  then []
  else (List.hd l) :: first_n (n-1) (List.tl l)

let rec part_n n firstl secondl =
  if n <= 0 || secondl = [] 
  then (firstl, secondl)
  else part_n (n-1) (firstl @ [List.hd secondl]) (List.tl secondl)

let rec maxl l = match l with
  | [] -> 0
  | a :: m -> max a (maxl m)

let rec minl l = match l with
  | [] -> 0
  | a :: m -> min a (minl m)

let rec mk_set l = match l with
    [] -> []
  | a :: m -> if List.mem a m then mk_set m else a :: mk_set m

let rec mk_set_fun f l = match l with
    [] -> []
  | a :: m -> 
      if List.mem (f a) (List.map f m) 
      then mk_set_fun f m 
      else a :: mk_set_fun f m

let list_diff l1 l2 = List.filter (fun x -> not (List.mem x l2)) l1

let list_inter l1 l2 = List.filter (fun x -> List.mem x l2) l1

let cartesian_product l l' = 
  List.concat (List.map (fun e -> List.map (fun e' -> (e,e')) l') l)

let rec rm_equal l = match l with
    []          -> []
  | [a]         -> [a]
  | a :: b :: m -> if a = b 
                   then rm_equal (b :: m)
                   else a :: rm_equal (b :: m) 

let rec mk_fast_set compare l = rm_equal (List.sort compare l)

let rec sum fl = if fl = [] then 0. else List.hd fl +. (sum (List.tl fl))

let rec topo_sort graph =
  let (topl,downl) = List.partition (fun (x,xl) -> xl = []) graph in
  match (topl,downl) with 
  | ([],[]) -> []
  | ([],_)  -> failwith "topo_sort: loop or missing nodes"
  | _       ->
    let topl' = List.map fst topl in
    let graph' = List.map (fun (x,xl) -> (x,list_diff xl topl')) downl in
    topl' @ topo_sort graph'

let rec is_prefix_list l1 l2 = match (l1,l2) with
    ([],_) -> true
  | (_,[]) -> false
  | (a :: m, a' :: m') -> a = a' && is_prefix_list m m'

let is_prefix s1 s2 = is_prefix_list (explode s1) (explode s2)

(*--------------------------------------------------------------------------
  Comparison
  -------------------------------------------------------------------------- *)

let compare_fst a b = compare (fst a) (fst b)
let compare_snd a b = compare (snd a) (snd b)
let compare_score a b = compare (snd b) (snd a)

(*--------------------------------------------------------------------------
  Hash
  -------------------------------------------------------------------------- *)

let values_of_hash hash = Hashtbl.fold (fun k v l -> v :: l) hash  []
let keys_of_hash hash = Hashtbl.fold (fun k v l -> k :: l) hash []
let alist_of_hash hash = Hashtbl.fold (fun k v l -> (k,v) :: l) hash  []

let hash_of_alist l = 
  let hash = Hashtbl.create (List.length l) in
  List.iter (fun (a,b) -> Hashtbl.add hash a b) l;
  hash

let hash_filter filter h =
  let f k v = if filter k v then () else Hashtbl.remove h k in
  Hashtbl.iter f (Hashtbl.copy h)

let hash_filter_new filter h =
  let h_new = Hashtbl.copy h in
  let f k v = if filter k v then () else Hashtbl.remove h_new k in
  Hashtbl.iter f h;
  h_new

let hash_map f h =
  let iter k v = Hashtbl.replace h k (f k v) in
  Hashtbl.iter iter (Hashtbl.copy h)

let hash_map_new f h =
  let h_new = Hashtbl.create (Hashtbl.length h) in
  let iter k v = Hashtbl.add h_new k (f k v) in
  Hashtbl.iter iter h;
  h_new

let hash_inv h = 
  hash_of_alist (List.map (fun (a,b) -> (b,a)) (alist_of_hash h)) 

let numberhash_of_list l =
  let h = Hashtbl.create (List.length l) in
  let i = ref 0 in
  let iter x = Hashtbl.add h x !i; incr i in
  List.iter iter l;
  h

let hash_init_of_keylist h f l =
  Hashtbl.clear h;
  let iter k = Hashtbl.add h k (f k) in
  List.iter iter l

let hash_set_append h k x =
  let oldl = try Hashtbl.find h k with _ -> [] in
  if List.mem x oldl then () 
    else Hashtbl.replace h k (x :: oldl)



(*--------------------------------------------------------------------------
  Naming
  -------------------------------------------------------------------------- *)

let fixed_constants = 
  ["$o"; "$t"; ">"; "!"; "?";"&";"|";"=>";"~";"=";"<=>";
   "$forall";"$exists";"$and";"$or";"$imply";"$imp";"$not";"$equals";
   "$true";"$false";"$i";"$tType"]

let logical_constants = 
  ["$o"; "$t"; ">"; "!"; "?";"&";"|";"=>";"~";"=";"<=>";
   "$true";"$false";"$i";"$tType"]

let is_lowercase c = 
  let i = Char.code c in i >= Char.code 'a' && i <= Char.code 'z' 
let is_uppercase c = 
  let i = Char.code c in i >= Char.code 'A' && i <= Char.code 'Z'
let is_numeric c =
  let i = Char.code c in i >= Char.code '0' && i <= Char.code '9'
let is_alphanumeric c =
  (is_lowercase c) or (is_uppercase c) or (is_numeric c) or 
  (Char.code c == Char.code '_')

let is_hhchar c =
  is_alphanumeric c 
  or (Char.code c == Char.code '\'')
  or (Char.code c == Char.code '/')
  or (Char.code c == Char.code '\\')  

let is_tptp s =
  let f c = if is_alphanumeric c then () else failwith "is_tptp" in
  can (String.iter f) s

let is_primed name =
  name.[0] = '\'' && name.[String.length name - 1] = '\''

let add_prime name = 
  if List.mem name fixed_constants || is_tptp name 
  then name
  else "\'" ^ name ^ "\'"

let remove_prime name =
  try 
  if is_primed name
  then 
    let n = String.length name in
    Str.last_chars (Str.first_chars name (n - 1)) (n- 2) 
  else name
  with _ -> name

(*--------------------------------------------------------------------------
  Address of libraries
  -------------------------------------------------------------------------- *)

let palibs_dir = "../../../hh2-data/palibs"
let data_dir = "../../../hh2-data"
let coq_dir = palibs_dir ^ "/" ^ "Coq-matching"
let h4_dir  = palibs_dir ^ "/" ^ "h4-kananaskis10/standard_library"
let hl_dir  = palibs_dir ^ "/" ^ "hl-225/standard_library"
let isa_dir = palibs_dir ^ "/" ^ "isa"
let mat_dir = palibs_dir ^ "/" ^ "matita/matita_out_new"
let miz_dir = palibs_dir ^ "/" ^ "mizar/xml2_out_bigger"
let miz_cj_dir = palibs_dir ^ "/" ^ "mizar_CICM_2016"
let chad_dir = palibs_dir ^ "/" ^ "chad_process"
let cakeml_dir = palibs_dir ^ "/" ^ "cakeml_2016_04_01"
let cakeml_new_dir = palibs_dir ^ "/" ^ "cakeml_2016_06_10"
let h4_ltb_dir  = palibs_dir ^ "/" ^ "h4-kananaskis10/ltb_2016_04_11"
let h4_ltb_new_dir = palibs_dir ^ "/" ^ "h4-kananaskis10/ltb_2016_06_10"

let lib_of s = match s with
    "coq" -> coq_dir
  | "h4"  -> h4_dir
  | "h4_ltb"  -> h4_ltb_dir
  | "h4_ltb_new"  -> h4_ltb_new_dir
  | "hl"  -> hl_dir
  | "isa" -> isa_dir
  | "mat" -> mat_dir
  | "miz" -> miz_dir
  | "miz_cj" -> miz_cj_dir
  | "chad" -> chad_dir
  | "cakeml" -> cakeml_dir
  | "cakeml_new" -> cakeml_new_dir
  | _     -> failwith "lib_of"

let mylib_of s = Str.string_after (lib_of s) 3


let prover_list = ["coq";"h4";"hl";"isa";"mat";"miz"]

let proverpairs_list = 
  let reorder (a,b) = if compare a b < 0 then (a,b) else (b,a) in
  let is_different (a,b) = a <> b in
  let l1 = cartesian_product prover_list prover_list in
  let l2 = List.filter is_different l1 in
  let l3 = mk_set (List.map reorder l2) in
  l3 

