structure hfAlign :> hfAlign =
struct

open HolKernel boolLib aiLib hfLib hfFormat hfImportSexp hfProofTerm hfTheory

val ERR = mk_HOL_ERR "hfAlign"

(* ------------------------------------------------------------------------
   String of types
   ------------------------------------------------------------------------ *)

fun name_vartype ty = "A" ^ (dest_vartype ty)
fun name_tyop (thy,tyop) = 
  if tyop = "bool" then "o"
  else if tyop = "set" then "i"
  else "t." ^ thy ^ "." ^ tyop

fun name_fo_fun_mono (s,f_arg,argl) =
  if null argl then s else
  (s ^ "(" ^ String.concatWith " " (map f_arg argl) ^ ")")

fun name_type ty =
  if is_vartype ty then name_vartype ty else
  let
    val {Args, Thy, Tyop} = dest_thy_type ty
    val tyops = name_tyop (Thy,Tyop)
  in
   if Tyop = "fun" 
   then "a" ^ String.concat (map name_type Args) 
   else  name_fo_fun_mono (tyops,name_type,Args)
  end

(* ------------------------------------------------------------------------
   Align types
   ------------------------------------------------------------------------ *)

fun hf_type ty = 
  let val {Args,Thy,Tyop} = dest_thy_type ty in
    case (Thy,Tyop,Args) of
      ("min","bool",[]) => Prop
    | ("min","fun",[a,b]) => TpArr (hf_type a, hf_type b)
    | ("hf",s,[]) => Base s
    | _ => raise ERR "hf_type" ""
  end

fun h4_type ty = case ty of
    Base s => mk_thy_type {Thy="hf",Tyop=s,Args=[]}
  | TpArr (ty1,ty2) => 
    let val (a,b) = (h4_type ty1, h4_type ty2) in 
      mk_funtype (a,b)
    end
  | Prop => bool

(* ------------------------------------------------------------------------
   Align terms
   ------------------------------------------------------------------------ *)

fun add_type (vs,ty) = vs ^ "_" ^ name_type ty
 
fun hf_const tm =
  let val {Thy,Name,Ty} = dest_thy_const tm in
    if Thy = "hf" orelse
       String.isPrefix "def_sk" Name orelse 
       String.isPrefix "DEF__" Name
    then Const Name else 
     Const (dfind tm h4cd) handle NotFound => 
     if term_eq tm implication then 
       let val (a,b) = (mk_var ("a",bool),mk_var ("b",bool)) in
         hf_term (list_mk_abs ([a,b],mk_imp (a,b)))
       end
     else if Thy = "bool" andalso Name = "!" then 
       let 
         val aty = #residue (singleton_of_list (
           match_type (type_of universal) (type_of tm)))
         val p = mk_var ("p",mk_funtype (aty,bool))
         val x = mk_var ("x",aty)
       in
         hf_term (mk_abs (p,mk_forall(x,mk_comb (p,x))))
       end
     else if Thy = "bool" andalso Name = "?" then 
       let 
         val aty = #residue (singleton_of_list (
           match_type (type_of existential) (type_of tm)))
         val p = mk_var ("p",mk_funtype (aty,bool))
         val x = mk_var ("x",aty)
       in
         hf_term (mk_abs (p,mk_exists(x,mk_comb (p,x))))
       end
     else if Thy = "min" andalso Name = "=" then 
       let 
         val aty = #residue (singleton_of_list (
           match_type (type_of equality) (type_of tm)))
         val x = mk_var ("x",aty)
         val y = mk_var ("y",aty)
       in
         hf_term (list_mk_abs ([x,y],mk_eq (x,y)))
       end
     else raise ERR "hf_const" (term_to_string tm)
  end
and hf_term tm =
  (
  if is_eq tm then 
    Eq (hf_type (type_of (lhs tm)), hf_term (lhs tm), hf_term (rhs tm))
  else if is_exists tm then 
    let 
      val (v,bod) = dest_exists tm 
      val (vs,ty) = dest_var v
    in
      Ex (add_type (vs,ty), hf_type ty, hf_term bod) 
    end
  else if is_forall tm then 
    let 
      val (v,bod) = dest_forall tm 
      val (vs,ty) = dest_var v
    in
      All (add_type (vs,ty), hf_type ty, hf_term bod)
    end
  else if is_imp_only tm then
    let val (a,b) = dest_imp tm in Imp (hf_term a, hf_term b) end
  else if is_abs tm then
    let 
      val (v,bod) = dest_abs tm 
      val (vs,ty) = dest_var v
    in
      Lam (add_type (vs,ty), hf_type ty, hf_term bod)
    end
  else if is_comb tm then 
    let val (a,b) = dest_comb tm in Ap (hf_term a, hf_term b) end
  else if is_var tm then 
    let val (vs,ty) = dest_var tm in
      Var (add_type (vs,ty)) 
    end
  else if is_const tm then hf_const tm
  else raise ERR "hf_term" "not supported"
  )
  handle HOL_ERR _ => (print_endline ("\n" ^ term_to_string tm); 
                       raise ERR "hf_term" "") 

fun h4_const s = 
  if dmem s hfcd 
    then dfind s hfcd 
  else if can prim_mk_const {Thy = "hf", Name = s}
    then prim_mk_const {Thy = "hf", Name = s}
  else raise ERR "h4_const" ""

fun h4_term_aux vl hftm = case hftm of
    Var s => valOf (List.find (fn x => fst (dest_var x) = s) vl)
  | Const s => h4_const s
  | Ap (t1,t2) => mk_comb (h4_term_aux vl t1, h4_term_aux vl t2)
  | Lam (s,ty,t) => 
    let val v = mk_var (s, h4_type ty) in
      mk_abs (v, h4_term_aux (v :: vl) t)
    end
  | Imp (t1,t2) => mk_imp (h4_term_aux vl t1, h4_term_aux vl t2)
  | All (s,ty,t) => 
    let val v = mk_var (s, h4_type ty) in
      mk_forall (v, h4_term_aux (v :: vl) t)
    end
  | Ex (s,ty,t) =>
    let val v = mk_var (s, h4_type ty) in
      mk_exists (v, h4_term_aux (v :: vl) t)
    end
  | Eq (_,t1,t2) => mk_eq (h4_term_aux vl t1, h4_term_aux vl t2)

fun h4_term hftm = h4_term_aux [] hftm
  
(* ------------------------------------------------------------------------
   Align proofs
   ------------------------------------------------------------------------ *)

fun hf_proof pftop = case pftop of
    KnO s => Known s
  | PrV i => PrVar (its i)
  | PrA (pf1,pf2) => PrAp (hf_proof pf1, hf_proof pf2)
  | TmA (pf,t) => TmAp (hf_proof pf, hf_term t)
  | PrL (i,t,pf) => PrLa (its i, hf_term t, hf_proof pf) 
  | TmL (v,pf) => 
     let val (vs,ty) = dest_var v in
       TmLa (vs ^ "_" ^ name_type ty, hf_type ty, hf_proof pf)
     end
  | ExT (ty1,ty2) => Ext (hf_type ty1,hf_type ty2)

end
