(* ========================================================================= *)
(* FILE          : hfLemScript.sml                                           *)
(* DESCRIPTION   : Useful lemmas for the HF theory                           *)
(* AUTHOR        : (c) Thibault Gauthier, Czech Technical University         *)
(* DATE          : 2020                                                      *)
(* ========================================================================= *)

open HolKernel boolLib aiLib hfLib boolTheory hfTheory 
  hfDep hfCheck hfDev hfExport

val ERR = mk_HOL_ERR "hfLemTheory"
(* load "hfLib"; load "hfTheory"; load "hfDev";
   open aiLib hfLib boolTheory hfTheory
   hfDep hfCheck hfDev hfExport; *)
val _ = new_theory "hfLem"

val _ = prooflog := [];
val (a,b) = (``a:bool``,``b:bool``);

(* ------------------------------------------------------------------------
   Anti-symmetry of implication
   ------------------------------------------------------------------------ *)

fun ANTISYM_IMP th1 th2 =
  let
    val p = mk_var ("p",list_mk_funtype ([bool,bool],bool));
    val template = ``!x0:bool x1:bool. p x0 x1 ==> (x0 = x1)``;
    val th5 = SUBST [p |-> ax15] template ax2;
    val th6 = CONV_RULE (TOP_DEPTH_CONV BETA_CONV) th5;
    val (t0,t1) = dest_imp (concl th1)
    val th3 = SPECL [t0,t1] th6
  in
    MP th3 (CONJ th1 th2)
  end

(* ------------------------------------------------------------------------
   Boolean cases
   ------------------------------------------------------------------------ *)

val boolcases = save_thm_export ("boolcases",
  let 
    val (x,y) = (mk_var ("x",bool),mk_var ("y",bool));
    val th1 = GEN y (DISCH y (ASSUME y));
    val th2 = EQ_MP (SYM True_def) th1;
    val th3 = ADD_ASSUM x (DISCH x th2);
    val th4 = DISCH T (ASSUME x);
    val th5 = ANTISYM_IMP th3 th4;
    val th6 = ASSUME ``~x``;
    val th7 = NOT_ELIM th6;
    val th8 = EQ_MP False_def (ASSUME F);
    val th9 = ADD_ASSUM ``~x`` th8;
    val th10 = SPEC x th9;
    val th11 = DISCH F th10;
    val th12 = ANTISYM_IMP th7 th11;
    val xthm = DISJ1 th5 ``x=F``;
    val nxthm = DISJ2 ``x=T`` th12;
    val disji = SPEC x xm;
    val disjo = DISJ_CASES disji xthm nxthm
  in
    GEN x disjo
  end);

(* ------------------------------------------------------------------------
   Adapted from boolTheory proofs.
   ------------------------------------------------------------------------ *)

val notall = save_thm_export ("notall",
  let
    val f = “_P:hf$set->bool”
    val x = “_x:hf$set”
    val t = mk_comb(f,x)
    val all = mk_forall(x,t)
    and exists = mk_exists(x,mk_neg t)
    val nott = ASSUME (mk_neg t)
    val th1 = CCONTR t (MP (ASSUME (mk_neg exists)) (EXISTS (exists,x) nott))
    val th2 = CCONTR exists (MP (ASSUME (mk_neg all)) (GEN x th1))
  in
    GEN f (DISCH (mk_neg all) th2)
  end
);

val notex = save_thm_export ("notex",
  let val f = “_P:hf$set->bool”
      val x = “_x:hf$set”
      val t = mk_comb(f,x)
      val tm = mk_neg(mk_exists(x,t))
      val all = mk_forall(x,mk_neg t)
      val asm1 = ASSUME t
      val thm1 = MP (ASSUME tm) (EXISTS (rand tm, x) asm1)
      val imp1 = DISCH tm (GEN x (NOT_INTRO (DISCH t thm1)))
  in
      GEN f imp1
  end);

val notconj = save_thm_export ("notconj", 
  let 
    val asm1 = ASSUME ``~(a /\ b)``
    val cnj = MP asm1 (CONJ (ASSUME a) (ASSUME b))
    val imp1 =
      let 
        val case1 = DISJ2 (mk_neg a) (NOT_INTRO (DISCH b cnj))
        val case2 = DISJ1 (ASSUME (mk_neg a)) (mk_neg b)
      in 
        DISJ_CASES (SPEC a xm) case1 case2
      end
  in
    GENL [a,b] (DISCH_ALL imp1)
  end);

val notdisj = save_thm_export ("notdisj", 
  let 
    val asm1 = ASSUME ``~(a \/ b)``
    val imp1 =
      let 
        val th1 = NOT_INTRO (DISCH a (MP asm1 (DISJ1 (ASSUME a) b)))
        val th2 = NOT_INTRO (DISCH b (MP asm1 (DISJ2 a (ASSUME b))))
      in 
        DISCH ``~(a \/ b)`` (CONJ th1 th2)
      end
  in
    GENL [a,b] imp1
  end);

val notnot = save_thm_export ("notnot",ax1);

val notnot_intro = save_thm_export ("notnot_intro",
  (GEN a (DISCH a (NOT_INTRO (DISCH (mk_neg a) 
    (UNDISCH (NOT_ELIM (ASSUME (mk_neg a)))))))));

val equivconj = save_thm_export ("equivconj", 
  let val (th1,th2) = EQ_IMP_RULE (ASSUME ``(a <=> b)``) in
    GENL [``a:bool``,``b:bool``]  (DISCH ``(a <=> b)`` (CONJ th1 th2))
  end);

(* ------------------------------------------------------------------------
   Swapping theorems
   ------------------------------------------------------------------------ *)

val swap1 = save_thm_export ("swap1",
  let 
    val imp = ``~a ==> ~b`` 
    val asm = DISCH b (CCONTR a (UNDISCH (UNDISCH (ASSUME imp))))
  in
    GENL [a,b] (DISCH imp asm)
  end);

val swap2 = save_thm_export ("swap2",
  let 
    val imp = ``~a ==> b``
    val asm1 = UNDISCH (ASSUME imp)
    val asm2 = UNDISCH (MP (SPEC b notnot_intro) asm1)
  in
    GENL [a,b] (DISCH imp (DISCH (mk_neg b) (CCONTR a asm2)))
  end);

val swap3 = save_thm_export ("swap3",
  let 
    val imp = ``a ==> ~b``
    val asm1 = UNDISCH (UNDISCH (ASSUME imp))
    val asm2 = NOT_INTRO (DISCH a asm1)
  in
    GENL [a,b] (DISCH imp (DISCH b asm2))
  end);

val swap4 = save_thm_export ("swap4", 
  let 
    val imp = ``a ==> b``
    val asm1 = UNDISCH (ASSUME imp)
    val asm2 = MP (SPEC b notnot_intro) asm1
    val asm3 = UNDISCH asm2
    val asm4 = NOT_INTRO (DISCH a asm3)
  in
    GENL [a,b] (DISCH imp (DISCH (mk_neg b) asm4))
  end);

val notequiv = save_thm_export ("notequiv", 
  let 
    val th1 = ASSUME ``a ==> b``
    val th2 = ASSUME ``b ==> a``
    val th3 = ANTISYM_IMP th1 th2
    val th4 = UNDISCH (MP (SPEC ``a <=> b`` notnot_intro) th3)   
    val th5 = DISCH ``b ==> a`` (NOT_INTRO (DISCH ``a ==> b`` th4))
  in
    GENL [``a:bool``,``b:bool``] (DISCH ``~(a <=> b)`` th5)
  end);

val notimp = save_thm_export ("notimp", 
  let 
    val asm1 = ASSUME a
    val asm2 = ASSUME (mk_neg a)
    val th1 = MP asm2 asm1
    val th2 = SPEC b (EQ_MP False_def th1)
    val th3 = DISCH (mk_neg a) (DISCH a th2)
    val (x1,x2) = dest_imp (concl th3)
    val th4 = MP (SPECL [dest_neg x1,x2] swap2) th3
    val conj1 = UNDISCH th4
    val th5 = DISCH b (DISCH a (ASSUME b))
    val (y1,y2) = dest_imp (concl th5)
    val th6 = MP (SPECL [y1,y2] swap4) th5
    val conj2 = UNDISCH th6
 in
   GENL [a,b] (DISCH_ALL (CONJ conj1 conj2))
 end);

val disjimp = save_thm_export ("disjimp",
  let 
    val asm1 = ASSUME a
    val asm2 = ASSUME (mk_neg a)
    val case1 = MP asm2 asm1
    val asm3 = ASSUME b
    val asm4 = ASSUME (mk_neg b)
    val case2 = MP asm4 asm3
    val disj = DISJ_CASES (ASSUME ``a \/ b``) case1 case2
  in
    GENL [a,b] (DISCH_ALL (DISCH (mk_neg a) (CCONTR b disj)))
  end);

val _ = export_theory ()
