structure hfMask :> hfMask =
struct

open HolKernel boolLib aiLib smlTimeout hfLib 
  hfProofRule hfProofTerm hfTheory

val ERR = mk_HOL_ERR "hfMask"

datatype defstatus = Def | Nodef | Fresh

datatype fot = 
    Forall of defstatus * term * fot
  | Exists of defstatus * term * fot
  | Conj of defstatus * fot * fot
  | Disj of defstatus * fot * fot
  | Imp of defstatus * fot * fot
  | Equiv of defstatus * fot * fot
  | Neg of defstatus * fot
  | Func of defstatus * fot * fot list
  | V of (defstatus * term)
  | C of (defstatus * term)

fun term_to_fot tm = 
  if is_forall tm then 
    let val (v,bod) = dest_forall tm in
      Forall (Fresh, v, term_to_fot bod)
    end
  else if is_exists tm then 
    let val (v,bod) = dest_exists tm in
      Exists (Fresh, v, term_to_fot bod)
    end
  else if is_conj tm then 
    let val (a,b) = dest_conj tm in
      Conj (Fresh, term_to_fot a, term_to_fot b)
    end 
  else if is_disj tm then 
    let val (a,b) = dest_disj tm in
      Disj (Fresh, term_to_fot a, term_to_fot b)
    end 
  else if is_imp_only tm then 
    let val (a,b) = dest_imp tm in
      Imp (Fresh, term_to_fot a, term_to_fot b)
    end 
  else if is_equiv tm then
    let val (a,b) = dest_eq tm in
      Equiv (Fresh, term_to_fot a, term_to_fot b)
    end
  else if is_neg tm then 
    let val a = dest_neg tm in
      Neg (Fresh, term_to_fot a)
    end
  else if is_comb tm then
    let val (oper,argl) = strip_comb tm in
      Func (Fresh, term_to_fot oper, map term_to_fot argl)
    end
  else if is_const tm then C (Fresh, tm)
  else if is_var tm then V (Def, tm)
  else (print_endline (term_to_string tm); raise ERR "term_to_fot" "")

fun is_fof tm = can term_to_fot tm

fun fot_to_term fot = case fot of
    Forall (s,t,f) => mk_forall (t, fot_to_term f)
  | Exists (s,t,f) => mk_exists (t, fot_to_term f)
  | Conj (s,f1,f2) => mk_conj (fot_to_term f1, fot_to_term f2)
  | Disj (s,f1,f2) => mk_disj (fot_to_term f1, fot_to_term f2)
  | Imp (s,f1,f2) => mk_imp (fot_to_term f1, fot_to_term f2)
  | Equiv (s,f1,f2) => mk_eq (fot_to_term f1, fot_to_term f2)
  | Neg (s,f) => mk_neg (fot_to_term f)
  | Func (s,f,fl) => list_mk_comb (fot_to_term f, map fot_to_term fl)
  | V (s,t) => t
  | C (s,t) => t

fun defstatus_of fot = case fot of
    Forall (s,t,f) => s 
  | Exists (s,t,f) => s
  | Conj (s,f1,f2) => s
  | Disj (s,f1,f2) => s
  | Imp (s,f1,f2) => s
  | Equiv (s,f1,f2) => s
  | Neg (s,f) => s
  | Func (s,f,fl) => s
  | V (s,t) => s
  | C (s,t) => s

fun add_status s1 s2 = case (s1,s2) of
    (_, Nodef) => Nodef
  | (Nodef,_) => Nodef
  | (Fresh, _) => Fresh
  | (_, Fresh) => Fresh
  | (Def,Def) => Def

fun add_statusl sl = case sl of
     [] => raise ERR "add_statusl" ""
   | [a] => a
   | a :: m => add_status a (add_statusl m)

fun backup fot =
  let 
    fun backup_binop cons f1 f2 =
      let val (f1',f2') = (backup f1, backup f2) in
        cons (add_status (defstatus_of f1') (defstatus_of f2'),f1',f2')
      end
  in
    case fot of
      Forall (s,t,f) => 
      let val f' = backup f in Forall (defstatus_of f',t,f') end
    | Exists (s,t,f) =>       
      let val f' = backup f in Exists (defstatus_of f',t,f') end 
    | Conj (s,f1,f2) => backup_binop Conj f1 f2
    | Disj (s,f1,f2) => backup_binop Disj f1 f2
    | Imp (s,f1,f2) => backup_binop Imp f1 f2
    | Equiv (s,f1,f2) => backup_binop Equiv f1 f2
    | Neg (s,f) => 
      let val f' = backup f in Neg (defstatus_of f',f') end
    | Func (s,f,fl) =>
      let 
        val fl' = map backup fl 
        val f' = backup f
        val news = add_status 
          (defstatus_of f') (add_statusl (map defstatus_of fl'))
      in 
        Func (news,f',fl')
      end
    | V (s,t) => fot 
    | C (s,t) => fot
  end       

fun decide_nextdef fot = 
  let 
    fun decide_binop cons s f1 f2 =
      if defstatus_of f1 = Fresh then 
        let val (f1a,f1b) = decide_nextdef f1 in
          (cons (s,f1a,f2), cons (s,f1b,f2))
        end
      else if defstatus_of f2 = Fresh then 
        let val (f2a,f2b) = decide_nextdef f2 in
          (cons (s,f1,f2a), cons (s,f1,f2b))
        end
      else raise ERR "decide_nextdef" "binop"
  in
    case fot of
      Forall (s,t,f) => 
      let val (fa,fb) = decide_nextdef f in 
        (Forall (s,t,fa), Forall (s,t,fb))
      end
    | Exists (s,t,f) =>       
      let val (fa,fb) = decide_nextdef f in 
        (Exists (s,t,fa), Exists (s,t,fb))
      end
    | Conj (s,f1,f2) => decide_binop Conj s f1 f2
    | Disj (s,f1,f2) => decide_binop Disj s f1 f2
    | Imp (s,f1,f2) => decide_binop Imp s f1 f2
    | Equiv (s,f1,f2) => decide_binop Equiv s f1 f2    
    | Neg (s,f) =>       
      let val (fa,fb) = decide_nextdef f in 
        (Neg (s,fa), Neg (s,fb))
      end
    | Func (s,f,fl) => (* arguments first *)
      if exists (fn x => defstatus_of x = Fresh) fl then
      let 
        val flag = ref false
        fun g x = 
          if (not (!flag)) andalso defstatus_of x = Fresh 
          then let val (a,b) = (decide_nextdef x) in flag := true; [a,b] end
          else [x]
        val (fla,flb) = pair_of_list (cartesian_productl (map g fl))
      in 
        (Func (s,f,fla), Func (s,f,flb))
      end
      else if defstatus_of f = Fresh then 
        let val (fa,fb) = decide_nextdef f in
          (Func (s,fa,fl), Func (s,fb,fl))
        end
      else raise ERR "decide_nextdef" "Func"
    | V (s,t) => raise ERR "decide_nextdef" "V"
    | C (s,t) => if s <> Fresh then raise ERR "decide_nextdef" "C"
                 else (C (Def,t), C (Nodef,t))
  end    

(* Definitions *)
val defi = ref 0;
val defmem = ref (dempty Term.compare)
val set = ``:hf$set``;

fun mk_def_atom tm = 
  let
    val vl = free_vars_lr tm
    val n = length vl
    val pty = list_mk_funtype ((List.tabulate (n, fn _ => set)), type_of tm)
    val pname = "_d" ^ its (!defi)
    val p = mk_var (pname, pty);
    val eq = mk_eq (p, list_mk_abs (free_vars_lr tm,tm))
    val _ = defmem := dadd (lhs eq) (rhs eq) (!defmem)
    val def = ASSUME eq
    val _ = incr defi
  in
    (lhs o concl) (expand_def def)
  end

fun loc_def fot = mk_def_atom (fot_to_term fot);

fun is_fotvar x = case x of V _ => true | _ => false;

fun mk_def_fot fot = 
  if defstatus_of fot = Def andalso not (is_fotvar fot)  
  then loc_def fot 
  else
  let fun fbinop mkf s f1 f2 = mkf (mk_def_fot f1, mk_def_fot f2) in
    case fot of
      Forall (s,t,f) => mk_forall (t,mk_def_fot f)
    | Exists (s,t,f) => mk_exists (t,mk_def_fot f)
    | Conj (s,f1,f2) => fbinop mk_conj s f1 f2
    | Disj (s,f1,f2) => fbinop mk_disj s f1 f2
    | Imp (s,f1,f2) => fbinop mk_imp s f1 f2
    | Equiv (s,f1,f2) => fbinop mk_eq s f1 f2
    | Neg (s,f) => mk_neg (mk_def_fot f)
    | Func (s,f,fl) => list_mk_comb (mk_def_fot f, map mk_def_fot fl)
    | V (s,t) => t
    | C (s,t) => t
  end;

(* Assumes fot is backed up already *)
fun gen_fot axl fot =
  if not (can decide_nextdef fot) then fot else
  let
    val (fa,fb) = decide_nextdef fot
    val (fa',fb') = (backup fa, backup fb)
  in
    if can (timeout 1.0 (METIS_PROVE axl)) (mk_def_fot fa') 
    then gen_fot axl fa' 
    else gen_fot axl fb'
  end

fun gen_cj axl cj = 
  let 
    val fot = backup (term_to_fot cj)
    val fotgen = gen_fot axl fot
  in  
    mk_def_fot fotgen
  end

fun ELIM_GENDEF th = 
  if is_forall (concl th) andalso 
     (String.isPrefix "_d" o fst o dest_var o fst o dest_forall o concl) th 
  then
    let 
      val v = (fst o dest_forall o concl) th
      val rep = dfind v (!defmem)
    in
      ELIM_GENDEF (SPEC rep th)
    end
  else th

end (* struct *)
