(* ===================================================================== *)
(* FILE         : hfProofRule.sml                                        *)
(* DESCRIPTION  : Proof terms for theorems                               *)
(*                This file follows the structure of std-thm.ML.         *)
(*                However, the primitive rules are different.            *)
(* AUTHOR       : Thibault Gauthier, Czech Technical University          *)
(* ===================================================================== *)

structure hfProofRule :> hfProofRule =
struct

open HolKernel boolLib boolTheory aiLib hfLib hfProofTerm normalForms

(*---------------------------------------------------------------------------
       Exception handling
 ---------------------------------------------------------------------------*)

type seq = hfProofTerm.pf * term set * term

val proof_err = mk_HOL_ERR "Proof"
fun ERR f m = raise proof_err f m
fun Assert b s1 s2 = if b then () else ERR s1 s2

(*---------------------------------------------------------------------------
    The following are here because I didn't want to Thm to be dependent
    on some derived syntax (now in boolSyntax).
 ---------------------------------------------------------------------------*)

fun seq_concl (_,_,c)  = c
fun seq_hypset (_,asl,_) = asl
fun seq_dest_thm (_,asl,w) = (asl,w)
fun var_occursl v l = isSome (HOLset.find (var_occurs v) l);
val empty_hyp = Term.empty_tmset
fun union_hyp asl1 asl2 = HOLset.union(asl1, asl2)

(*---------------------------------------------------------------------------*
 * The type of pre-theorems and some basic operations on it.                 *
 *---------------------------------------------------------------------------*)

val last_rules = ref []
fun catch msg f x = (last_rules := msg :: !last_rules; f x)

fun pfKnO s = KnO s

(* should contain both free variables and free proof variables *)
fun pfASSUME_err s = 
  (PrV (gen_pfv s), HOLset.singleton Term.compare s,s)
fun pfASSUME s = catch "pfASSUME" pfASSUME_err s

fun pfDISCH_err s (pf,asl,w) = 
  (PrL (gen_pfv s, s, pf), 
   HOLset.delete (asl,s) handle NotFound => asl, 
   mk_imp (s,w))
fun pfDISCH s (pf,asl,w) = catch "pfDISCH" (pfDISCH_err s) (pf,asl,w)

fun pfGEN_err v (pf,asl,w) = 
  (Assert (not (var_occursl v asl)) "pfGEN" "";
   (TmL (v,pf), asl, mk_forall (v,w)))
fun pfGEN v (pf,asl,w) = catch "pfGEN" (pfGEN_err v) (pf,asl,w)

fun pfSPEC_err t (pf,asl,w) =
  let 
   val (Rator,Rand) = dest_comb w
   val {Thy,Name,...} = dest_thy_const Rator
  in
   Assert (Name="!" andalso Thy="bool") "pfSPEC" "";
   (TmA (pf,t), asl, beta_conv(mk_comb(Rand, t)))
  end
fun pfSPEC t (pf,asl,w) = catch "pfSPEC" (pfSPEC_err t) (pf,asl,w)

fun pfSpecialize_err t (pf,asl,w) =
  let 
    val (Rator,Rand) = dest_comb w
    val {Name,Thy,...} = dest_thy_const Rator
  in
    Assert (Name="!" andalso Thy="bool") "pfSpecialize" "";
    (TmA (pf,t), asl, beta_conv(mk_comb(Rand,t)))
  end
fun pfSpecialize t (pf,asl,w) = 
  catch "pfSpecialize" (pfSpecialize_err t) (pf,asl,w)

fun pfEXT_err (a,b) =
  let 
    val ab = mk_funtype (a,b)
    val (f,g,x) = (genvar ab, genvar ab, genvar a)
    val eqfg = mk_eq (f,g)
    val eqfgx = mk_forall (x, (mk_eq (mk_comb (f,x), mk_comb (g,x))))
    val imp = mk_imp (eqfgx,eqfg)
    val finaltm = mk_forall (f, mk_forall (g,imp))
  in
    (ExT (a,b), empty_hyp, finaltm)
  end
fun pfEXT (a,b) = catch "pfEXT" pfEXT_err (a,b)

(* Uses this only if t is beta-eta convertible to w *) 
fun reduce_once t = 
  if can beta_conv t then beta_conv t
  else if can eta_conv t then eta_conv t
  else if is_comb t then 
    let val (a,b) = dest_comb t in
      mk_comb (reduce_once a, reduce_once b)
    end
  else if is_abs t then 
    let val (v,bod) = dest_abs t in
      mk_abs (v, reduce_once bod)
    end
  else t

fun normalize t = 
  let val t1 = reduce_once t in
    if aconv t1 t then t else normalize t1
  end

fun convertible t1 t2 = aconv (normalize t1) (normalize t2)

fun pfCONVERT_nocheck t (pf,asl,w) = (pf,asl,t)
(* if convertible t w 
   then (pf,asl,t)
   else (print "Error: not convertible\n"; raise ERR "pfCONVERT" "")
*)

fun pfBeta_err (pf,asl,w) = 
  let val (a,b) = dest_eq w in 
    (pf, asl, mk_eq (a, beta_conv b))
  end
fun pfBeta (pf,asl,w) = catch "pfBeta" pfBeta_err (pf,asl,w) 

fun pfMP_ONLY_err (pf1,asl1,w1) (pf2,asl2,w2) = 
  let val (ant,conseq) = dest_imp w1 in
    Assert (aconv ant w2) "pfMP_ONLY" "not alpha-convertible";
    (PrA (pf1,pf2), union_hyp asl1 asl2, conseq)  
  end
fun pfMP_ONLY (pf1,asl1,w1) (pf2,asl2,w2) =
  catch "pfMP_ONLY" (pfMP_ONLY_err (pf1,asl1,w1)) (pf2,asl2,w2)

(* ------------------------------------------------------------------------
   Definition of equality and existential quantifier.
   ------------------------------------------------------------------------ *)

fun pfEQ_FOLD_err (pf,asl,w) =
  let 
    val imp = snd (dest_forall w)
    val s = rand (snd (dest_imp imp)) 
    val t = rand (fst (dest_imp imp))
  in
    (pf, asl, mk_eq (s,t))
  end
fun pfEQ_FOLD (pf,asl,w) = catch "pfEQ_FOLD" pfEQ_FOLD_err (pf,asl,w)

fun pfEQ_UNFOLD_err (pf,asl,w) =   
  let 
    val (s,t) = dest_eq w
    val sty = type_of s
    val qty = list_mk_funtype ([sty,sty],bool)
    val q = genvar qty
    val imp = mk_imp (list_mk_comb (q,[s,t]), list_mk_comb (q,[t,s]))
  in
    (pf,asl,mk_forall (q,imp))
  end
fun pfEQ_UNFOLD (pf,asl,w) = catch "pfEQ_UNFOLD" pfEQ_UNFOLD_err (pf,asl,w)

fun pfEX_FOLD_err (pf,asl,w) =
  let 
    val (_,bod1) = dest_forall w
    val (x,bod2) = dest_forall (fst (dest_imp bod1))
    val qx = fst (dest_imp bod2)
  in
    (pf,asl, mk_exists (x,qx))
  end
fun pfEX_FOLD (pf,asl,w) = catch "pfEX_FOLD" pfEX_FOLD_err (pf,asl,w)

fun pfEX_UNFOLD_err (pf,asl,w) = 
  let
    val (x,qx) = dest_exists w
    val p = genvar bool
    val imp = mk_imp (mk_forall (x,mk_imp (qx,p)), p)
  in
    (pf,asl, mk_forall (p,imp))
  end
fun pfEX_UNFOLD (pf,asl,w) = catch "pfEX_UNFOLD" pfEX_UNFOLD_err (pf,asl,w) 

fun pfINST_NOHYP_err theta (pf,asl,w) =
  (
  Assert (List.all (is_var o #redex) theta) "pfINST_NOHYP" "not a var";
  Assert (HOLset.numItems asl = 0) "pfINST_NOHYP" "not empty hyp";
  (pfsubst_term theta pf, asl, subst theta w)
  )
fun pfINST_NOHYP theta (pf,asl,w) =
  catch "pfINST_NOHYP" (pfINST_NOHYP_err theta) (pf,asl,w)

fun pfINST_TYPE_NOHYP_err theta (pf,asl,w) =
  (
  Assert (HOLset.numItems asl = 0) "pfINST_TYPE_NOHYP" "";
  (pfsubst_type theta pf, asl, inst theta w)
  )
fun pfINST_TYPE_NOHYP theta (pf,asl,w) =
  catch "pfINST_TYPE_NOHYP" (pfINST_TYPE_NOHYP_err theta) (pf,asl,w)

(* ------------------------------------------------------------------------
   DERIVED RULES
   ------------------------------------------------------------------------ *)

(* ------------------------------------------------------------------------
   Hypothesis rules
   ------------------------------------------------------------------------ *)

fun pfDISCHL tml seq = case tml of
    [] => seq
  | a :: m => pfDISCHL m (pfDISCH a seq)

fun pfUNDISCH_err (seq as (_,_,w)) =
  let val (t,_) = dest_imp w in pfMP_ONLY seq (pfASSUME t) end
fun pfUNDISCH seq = catch "pfUNDISCH" pfUNDISCH_err seq

fun pfUNDISCHN n seq = funpow n pfUNDISCH seq

(* ------------------------------------------------------------------------
   Quantifiers rules
   ------------------------------------------------------------------------ *)

fun pfGENL vl seq = case vl of
    [] => seq 
  | v :: m => pfGEN v (pfGENL m seq) 

fun pfEXISTS_err (extm,t) seq =
  let 
    val (x,qx) = dest_exists extm
    val p = genvar bool
    val assum = mk_forall (x, mk_imp (qx,p))
    val seq2 = pfASSUME assum
    val seq3 = pfSPEC t seq2
    val seq4 = pfMP_ONLY seq3 seq
  in
    pfEX_FOLD (pfGEN p (pfDISCH assum seq4))
  end
fun pfEXISTS (extm,t) seq = catch "pfEXISTS" (pfEXISTS_err (extm,t)) seq

fun pfCHOOSE_err (v,(seq1 as (_,_,w1))) (seq2 as (_,_,w2)) =
  let 
    val (x,qx) = dest_exists w1
    val qv = subst [{redex = x,residue = v}] qx
    val seq1' = pfSPEC w2 (pfEX_UNFOLD seq1)
    val seq2' = pfGEN v (pfDISCH qv seq2)
  in
    pfMP_ONLY seq1' seq2'
  end
fun pfCHOOSE (v,seq1) seq2 = catch "pfCHOOSE" (pfCHOOSE_err (v,seq1)) seq2

fun pfINST_err theta (seq as (_,asl,_)) = 
  let
    val seq1 = pfDISCHL (HOLset.listItems asl) seq 
    val seq2 = pfINST_NOHYP theta seq1
  in
    pfUNDISCHN (HOLset.numItems asl) seq2
  end
fun pfINST theta seq = catch "pfINST" (pfINST_err theta) seq

fun pfINST_TYPE_err theta (seq as (_,asl,_)) = 
  let 
    val seq1 = pfDISCHL (HOLset.listItems asl) seq 
    val seq2 = pfINST_TYPE_NOHYP theta seq1
  in
    pfUNDISCHN (HOLset.numItems asl) seq2
  end
fun pfINST_TYPE theta seq = 
  catch "pfINST_TYPE" (pfINST_TYPE_err theta) seq  

(* ------------------------------------------------------------------------
   Shortands for writing definitional theorems
   ------------------------------------------------------------------------ *)

fun ktm tm = (KnO "D", HOLset.empty Term.compare, tm);

fun pfINST_TY t seq = 
  let 
    val ty = type_of t
    val theta = [{redex = alpha, residue = ty}] 
  in
    pfINST_TYPE_NOHYP theta seq
  end

fun pfSPECL tml seq = case tml of
    [] => seq
  | a :: m => pfSPECL m (pfSPEC a seq)

(* ------------------------------------------------------------------------
   Reflexivity
   ------------------------------------------------------------------------ *)

fun pfREFL_err t =
  let 
    val a = type_of t
    val qty = list_mk_funtype ([a,a],bool)
    val q = genvar qty
    val qtt = list_mk_comb (q,[t,t])
    val seq1 = pfASSUME qtt
    val seq2 = pfDISCH qtt seq1
    val seq3 = pfGEN q seq2
  in
    pfEQ_FOLD seq3
  end
fun pfREFL t = catch "pfREFL" pfREFL_err t

val lemREFL = ktm ``!_x:'a. _x = _x``;
fun shREFL t = pfSPEC t (pfINST_TY t lemREFL)

fun pfBETA_CONV t = 
  let val t' = beta_conv t in
    pfCONVERT_nocheck (mk_eq (t,t')) (pfREFL t)
  end
fun shBETA_CONV t = 
  let val t' = beta_conv t in
    pfCONVERT_nocheck (mk_eq (t,t')) (shREFL t)
  end

fun pfALPHA t1 t2 = pfCONVERT_nocheck (mk_eq (t1,t2)) (pfREFL t1)
fun shALPHA t1 t2 = pfCONVERT_nocheck (mk_eq (t1,t2)) (shREFL t1)

val pfETA_AX = 
  let val t = fst (dest_forall (concl ETA_AX)) in
    pfCONVERT_nocheck (concl ETA_AX) (pfGEN t (pfREFL t)) 
  end
val shETA_AX = ktm (concl ETA_AX)

(* ------------------------------------------------------------------------
   Rewriting with equality
   ------------------------------------------------------------------------ *)

fun pfLEIBNIZ_err (seq as (pf,asl,w)) =
  let 
    val xty = type_of (lhs w)
    val pty = mk_funtype (xty, bool)
    val (x,y,p) = (genvar xty,genvar xty,genvar pty)
    val seq2 = pfEQ_UNFOLD seq
    val lam = mk_abs (x, mk_abs (y, mk_comb (p,x)))
    val seq3 = pfSPEC lam seq2
    val norm = (mk_imp (mk_comb (p, lhs w), mk_comb (p,rhs w)))
    val seq4 = pfCONVERT_nocheck norm seq3
  in
    pfGEN p seq4
  end
fun pfLEIBNIZ seq = catch "pfLEIBNIZ" pfLEIBNIZ_err seq
fun shLEIBNIZ seq = 
  let val (_,asl,w) = pfLEIBNIZ seq in (* hides INST_TYPE *)
    (KnO "D",asl,w)
  end


fun pfSUBST_ONE_err (v,template) (seqeq as (pfeq,asleq,weq)) 
  (seq as (pf,asl,w)) =
  let
    val seq1 = pfLEIBNIZ seqeq
    val abs = mk_abs (v,template)
    val seq2 = pfSPEC abs seq1
    val norm1 = beta_conv (mk_comb (abs,lhs weq))
    val norm2 = beta_conv (mk_comb (abs,rhs weq))
    val norm = mk_imp (norm1,norm2)
    val seq3 = pfCONVERT_nocheck norm seq2
  in
    pfMP_ONLY seq3 seq
  end
fun pfSUBST_ONE (v,template) seqeq seq =
  catch "pfSUBST_ONE" (pfSUBST_ONE_err (v,template) seqeq) seq
fun shSUBST_ONE (v,template) (seqeq as (pfeq,asleq,weq)) 
  (seq as (pf,asl,w)) =
  let
    val seq1 = shLEIBNIZ seqeq
    val abs = mk_abs (v,template)
    val seq2 = pfSPEC abs seq1
    val norm1 = beta_conv (mk_comb (abs,lhs weq))
    val norm2 = beta_conv (mk_comb (abs,rhs weq))
    val norm = mk_imp (norm1,norm2)
    val seq3 = pfCONVERT_nocheck norm seq2
  in
    pfMP_ONLY seq3 seq
  end


fun pfLEIBNIZL_err eql =
  let 
    val (lhsl,rhsl) = (map lhs (map #3 eql), map rhs (map #3 eql))
    val (lhsv,rhsv) = (Vector.fromList lhsl, Vector.fromList rhsl)
    val pty = list_mk_funtype (map type_of lhsl, bool)
    val p = genvar pty
    val ptm = list_mk_comb (p,lhsl)
    val seqa = pfASSUME ptm
    val xl = map (fn x => genvar (type_of x)) lhsl
    val xv = Vector.fromList xl
    fun fv xi i = if i < xi then rhsv else if i = xi then xv else lhsv
    val n = Vector.length xv
    val laml1 = List.tabulate (n, fn xi => 
      List.tabulate (n, fn i => Vector.sub (fv xi i,i)))
    fun fx i l = (Vector.sub (xv,i), list_mk_comb (p,l))
    val laml2 = mapi fx laml1 
    val lameql = combine (laml2,eql)
    fun loop seq l = case l of
        [] => seq
      | (alam,aseq) :: m => loop (pfSUBST_ONE alam aseq seq) m
  in  
    pfGEN p (pfDISCH ptm (loop seqa lameql))
  end
fun pfLEIBNIZL eql = catch "pfLEIBNIZ" pfLEIBNIZL_err eql
fun shLEIBNIZL_err eql =
  let 
    val (lhsl,rhsl) = (map lhs (map #3 eql), map rhs (map #3 eql))
    val (lhsv,rhsv) = (Vector.fromList lhsl, Vector.fromList rhsl)
    val pty = list_mk_funtype (map type_of lhsl, bool)
    val p = genvar pty
    val ptm = list_mk_comb (p,lhsl)
    val seqa = pfASSUME ptm
    val xl = map (fn x => genvar (type_of x)) lhsl
    val xv = Vector.fromList xl
    fun fv xi i = if i < xi then rhsv else if i = xi then xv else lhsv
    val n = Vector.length xv
    val laml1 = List.tabulate (n, fn xi => 
      List.tabulate (n, fn i => Vector.sub (fv xi i,i)))
    fun fx i l = (Vector.sub (xv,i), list_mk_comb (p,l))
    val laml2 = mapi fx laml1 
    val lameql = combine (laml2,eql)
    fun loop seq l = case l of
        [] => seq
      | (alam,aseq) :: m => loop (shSUBST_ONE alam aseq seq) m
  in  
    pfGEN p (pfDISCH ptm (loop seqa lameql))
  end
fun shLEIBNIZL eql = catch "shLEIBNIZ" shLEIBNIZL_err eql

fun pfSUBST_err oldsubst template seq = 
  let
    val fvs = Term.FVL [template] Term.empty_varset
    val subst = filter (fn {redex,residue} => HOLset.member (fvs,redex)) 
      oldsubst
  in
    case subst of
    [] => seq
  | [{redex,residue}] => pfSUBST_ONE (redex,template) residue seq
  | _ =>
    let 
      val eql = map #residue subst
      val vl = map #redex subst
      val lam = list_mk_abs (vl,template)
      val seq1 = pfLEIBNIZL eql
      val seq2 as (_,_,w2) = pfSPEC lam seq1 
      val (a,b) = dest_imp w2
      val an = list_beta_conv a
      val bn = list_beta_conv b
      val t = mk_imp (an,bn)
      val seq3 = pfCONVERT_nocheck t seq2
    in
      pfMP_ONLY seq3 seq
    end
  end
fun pfSUBST subst template seq =
  catch "pfSUBST" (pfSUBST_err subst template) seq
fun shSUBST_err oldsubst template seq = 
  let
    val fvs = Term.FVL [template] Term.empty_varset
    val subst = filter (fn {redex,residue} => HOLset.member (fvs,redex)) 
      oldsubst
  in
    case subst of
    [] => seq
  | [{redex,residue}] => shSUBST_ONE (redex,template) residue seq
  | _ =>
    let 
      val eql = map #residue subst
      val vl = map #redex subst
      val lam = list_mk_abs (vl,template)
      val seq1 = shLEIBNIZL eql
      val seq2 as (_,_,w2) = pfSPEC lam seq1 
      val (a,b) = dest_imp w2
      val an = list_beta_conv a
      val bn = list_beta_conv b
      val t = mk_imp (an,bn)
      val seq3 = pfCONVERT_nocheck t seq2
    in
      pfMP_ONLY seq3 seq
    end
  end
fun shSUBST subst template seq =
  catch "shSUBST" (shSUBST_err subst template) seq

fun pfEQ_MP_err (eq as (_,_,w)) seq =
  let 
    val v = genvar (type_of (lhs w)) 
    val subst = [{redex=v,residue=eq}]
    val template = v
  in
    pfSUBST subst template seq
  end
fun pfEQ_MP eq seq = catch "pfEQ_MP"  (pfEQ_MP_err eq) seq
val lemEQ_MP = ktm ``!_a _b. ((_a <=> _b) /\ _a) ==> _b``;



fun pfAP_TERM_err tm (eq as (_,_,w)) =
  let 
    val t1 = lhs w
    val seq = pfREFL (mk_comb (tm,t1))
    val v = genvar (type_of t1)
    val subst = [{redex=v,residue=eq}]
    val template = mk_eq (mk_comb (tm,t1), mk_comb (tm,v))
  in
    pfSUBST subst template seq
  end
fun pfAP_TERM tm eq = catch "pfAP_TERM" (pfAP_TERM_err tm) eq
val lemAP_TERM = ktm ``!_f _x _y. ((_x :'a) = _y) ==> (_f _x :'b = _f _y)``;
fun shAP_TERM_err tm (eq as (_,_,w)) =
  let 
    val (x,y) = dest_eq w
    val (tya,tyb) = dom_rng (type_of tm)
    val theta = [alpha |-> tya, beta |-> tyb]
    val seq1 = pfINST_TYPE_NOHYP theta lemAP_TERM
  in
    pfMP_ONLY (pfSPECL [tm,x,y] seq1) eq
  end
fun shAP_TERM tm eq = catch "shAP_TERM" (shAP_TERM_err tm) eq

fun pfAP_THM_err (eq as (_,_,w)) tm =
  let 
    val t1 = lhs w
    val seq = pfREFL (mk_comb (t1,tm))
    val v = genvar (type_of t1)
    val subst = [{redex=v,residue=eq}]
    val template = mk_eq (mk_comb (t1,tm), mk_comb (v,tm))
  in
    pfSUBST subst template seq
  end
fun pfAP_THM eq tm = catch "pfAP_THM" (pfAP_THM_err eq) tm
val lemAP_THM = ktm ``!_f _g _x:'a. _f = _g ==> (_f _x :'b = _g _x)``;
fun shAP_THM_err (eq as (_,_,w)) tm =
  let 
    val (f,g) = dest_eq w
    val (tya,tyb) = dom_rng (type_of f)
    val theta = [alpha |-> tya, beta |-> tyb]
    val seq1 = pfINST_TYPE_NOHYP theta lemAP_THM
  in
    pfMP_ONLY (pfSPECL [f,g,tm] seq1) eq
  end
fun shAP_THM eq tm = catch "shAP_THM" (shAP_THM_err eq) tm

fun pfSYM_err (eq as (_,_,w)) =
  let 
    val t1 = lhs w
    val seq = pfREFL t1
    val v = genvar (type_of t1)
    val subst = [{redex=v,residue=eq}]
    val template = mk_eq (v,t1)
  in
    pfSUBST subst template seq
  end
fun pfSYM eq = catch "pfSYM" pfSYM_err eq
val lemSYM = ktm ``!_x:'a _y. _x = _y ==> _y = _x``;
fun shSYM_err (eq as (_,_,w)) = 
  let val (x,y) = dest_eq w in
    pfMP_ONLY (pfSPECL [x,y] (pfINST_TY x lemSYM)) eq
  end
fun shSYM eq = catch "shSYM" shSYM_err eq

fun pfTRANS_err (eq1 as (_,_,w1)) eq2 =
  let 
    val t1 = lhs w1
    val v = genvar (type_of t1)
    val subst = [{redex=v,residue=eq2}]
    val template = mk_eq (t1,v)
  in
    pfSUBST subst template eq1
  end
fun pfTRANS eq1 eq2 = catch "pfTRANS" (pfTRANS_err eq1) eq2
val lemTRANS = ktm ``!_x:'a _y _z. (_x = _y /\ _y = _z) ==> _x = _z``;



fun pfMK_COMB_err (funeq as (_,_,funw)) (argeq as (_,_,argw)) =
  let
    val (f,g) = dest_eq funw
    val (x,y) = dest_eq argw
    val seq = pfREFL (mk_comb (f,x))
    val vf = genvar (type_of f)
    val vx = genvar (type_of x)
    val template = mk_eq (mk_comb (f,x), mk_comb (vf,vx))
    val subst = [{redex=vf,residue=funeq},{redex=vx,residue=argeq}]
  in
    pfSUBST subst template seq
  end
fun pfMK_COMB funeq argeq = 
  catch "pfMK_COMB" (pfMK_COMB_err funeq) argeq
val lemMK_COMB = ktm 
  ``!_f _g _x _y. ((_f:'a -> 'b) = _g /\ (_x:'a = _y)) ==> _f _x = _g _y``;


fun pfEQ_IMP_RULE_LEFT_err (eq as (_,_,w)) =
  let val (t1,t2) = dest_eq w in 
    pfDISCH t1 (pfEQ_MP eq (pfASSUME t1))
  end
fun pfEQ_IMP_RULE_LEFT eq = 
  catch "pfEQ_IMP_RULE_LEFT" pfEQ_IMP_RULE_LEFT_err eq


fun pfEQ_IMP_RULE_RIGHT_err (eq as (_,_,w)) =
  let val (t1,t2) = dest_eq w in
    pfDISCH t2 (pfEQ_MP (pfSYM eq) (pfASSUME t2))
  end
fun pfEQ_IMP_RULE_RIGHT eq = 
  catch "pfEQ_IMP_RULE_RIGHT" pfEQ_IMP_RULE_RIGHT_err eq





(* ------------------------------------------------------------------------
   Quantifiers and equality
   ------------------------------------------------------------------------ *)

fun pfABS_err v (eq as (_,_,w)) =
  let
    val (t1,t2) = dest_eq w
    val (aty,bty) = (type_of v, type_of t1)
    val (t1lam,t2lam) = (mk_abs (v,t1), mk_abs (v,t2))
    val (t1',t2') = (mk_comb (t1lam,v), mk_comb (t2lam,v))
    val eq1 = pfCONVERT_nocheck (mk_eq (t1',t2')) eq
    val eq2 = pfGEN v eq1
    val imp1 = pfEXT (aty,bty)
    val imp2 = pfSPEC t2lam (pfSPEC t1lam imp1) 
  in
    pfMP_ONLY imp2 eq2
  end
fun pfABS v eq = catch "pfABS" (pfABS_err v) eq

fun pfGEN_ABS_err termo vl seq = case vl of
    [] => seq
  | v :: m => 
    let val seq1 as (_,_,w1) = pfABS v (pfGEN_ABS_err termo m seq) in
      if not (isSome termo) then seq1 else 
      let 
        val binder = valOf termo 
        val ty2 = mk_funtype (type_of (lhs w1), bool)
        val ty1 = type_of binder
        val theta = match_type ty1 ty2
        val binderi = inst theta binder
     in
       pfAP_TERM binderi seq1
     end
   end
fun pfGEN_ABS termo vl seq =
  catch "pfGEN_ABS" (pfGEN_ABS_err termo vl) seq

(* ------------------------------------------------------------------------
   Rules for logical connectives
   ------------------------------------------------------------------------ *)

open hfTheory

val logicthml = 
  filter (fn x => not (String.isPrefix "ax" (fst x))) (DB.thms "hf")

val logicthmd = dnew String.compare logicthml

fun fetch_seq name = 
  let 
    val thm = dfind name logicthmd
    val asl = hypset thm 
    val w = concl thm
  in
    (KnO name,asl,w)
  end

fun pfCONJ_err (seq1 as (_,_,w1)) (seq2 as (_,_,w2)) =
  let
    val andI = fetch_seq "andI"
    val seq3 = pfSPEC w2 (pfSPEC w1 andI)
  in
    pfMP_ONLY (pfMP_ONLY seq3 seq1) seq2
  end
fun pfCONJ seq1 seq2 = catch "pfCONJ" (pfCONJ_err seq1) seq2

fun pfCONJUNCT1_err (seq as (_,_,w)) =
  let
    val andEL = fetch_seq "andEL"
    val (t1,t2) = dest_conj w
  in
    pfMP_ONLY (pfSPEC t2 (pfSPEC t1 andEL)) seq
  end
fun pfCONJUNCT1 seq = catch "pfCONJUNCT1" pfCONJUNCT1_err seq

fun pfCONJUNCT2_err (seq as (_,_,w)) =
  let 
    val andER = fetch_seq "andER"
    val (t1,t2) = dest_conj w
  in
    pfMP_ONLY (pfSPEC t2 (pfSPEC t1 andER)) seq
  end
fun pfCONJUNCT2 seq = catch "pfCONJUNCT2" pfCONJUNCT2_err seq

fun pfDISJ1_err (seq as (_,_,w)) t2 =
  let val orIL = fetch_seq "orIL" in
    pfMP_ONLY (pfSPEC t2 (pfSPEC w orIL)) seq
  end
fun pfDISJ1 seq t2 = catch "pfDISJ1" (pfDISJ1_err seq) t2

fun pfDISJ2_err t1 (seq as (_,_,w)) =
  let val orIR = fetch_seq "orIR" in
    pfMP_ONLY (pfSPEC w (pfSPEC t1 orIR)) seq
  end
fun pfDISJ2 t1 seq = catch "pfDISJ2" (pfDISJ2_err t1) seq

fun pfDISJ_CASES_err (seq1 as (_,_,w1)) (seq2 as (_,_,w2)) seq3 =
  let
    val orE = fetch_seq "orE"
    val (t1,t2) = dest_disj w1
    val seq4 = pfSPEC t2 (pfSPEC t1 orE)
    val seq5 = pfSPEC w2 (pfMP_ONLY seq4 seq1)
  in
    pfMP_ONLY (pfMP_ONLY seq5 (pfDISCH t1 seq2)) (pfDISCH t2 seq3) 
  end
fun pfDISJ_CASES seq1 seq2 seq3 = 
  catch "pfDISJ_CASES" (pfDISJ_CASES_err seq1 seq2) seq3

fun pfNOT_INTRO_err (seq as (_,_,w)) =
  let
    val notI = fetch_seq "notI"
    val (t,_) = dest_imp w
  in 
    pfMP_ONLY (pfSPEC t notI) seq
  end
fun pfNOT_INTRO seq = catch "pfNOT_INTRO" pfNOT_INTRO_err seq

fun pfNOT_ELIM_err (seq as (_,_,w)) =
  let
    val notE = fetch_seq "notE"
    val t = rand w
  in 
    pfMP_ONLY (pfSPEC t notE) seq
  end
fun pfNOT_ELIM seq = catch "pfNOT_ELIM" pfNOT_ELIM_err seq

fun pfMP_err (seq1 as (_,_,w1)) seq2 = 
  if is_neg w1
  then pfMP_ONLY (pfNOT_ELIM seq1) seq2
  else pfMP_ONLY seq1 seq2
fun pfMP seq1 seq2 = catch "pfMP" (pfMP_err seq1) seq2

fun pfCCONTR_err t seq = 
  let 
    val contra = fetch_seq "contra"
    val nt = mk_neg t
  in 
    pfMP_ONLY (pfSPEC t contra) (pfDISCH nt seq)
  end
fun pfCCONTR t seq = catch "pfCCONTR" (pfCCONTR_err t) seq

(* *)
fun shEQ_MP_err (eq as (_,_,w)) seq = 
  let val (a,b) = dest_eq w in
    pfMP_ONLY (pfSPECL [a,b] lemEQ_MP) (pfCONJ eq seq)
  end
fun shEQ_MP eq seq = catch "shEQ_MP"  (shEQ_MP_err eq) seq
 
fun shTRANS_err (eq1 as (_,_,w1)) (eq2 as (_,_,w2)) = 
  let val (x,y) = dest_eq w1 in
    pfMP_ONLY (pfSPECL [x,y,rhs w2] (pfINST_TY x lemTRANS)) (pfCONJ eq1 eq2)
  end
fun shTRANS eq1 eq2 = catch "shTRANS" (shTRANS_err eq1) eq2

fun shMK_COMB_err (funeq as (_,_,funw)) (argeq as (_,_,argw)) =
  let     
    val (f,g) = dest_eq funw
    val (x,y) = dest_eq argw
    val (tya,tyb) = dom_rng (type_of f)
    val theta = [alpha |-> tya, beta |-> tyb]
    val seq1 = pfINST_TYPE_NOHYP theta lemMK_COMB
  in
    pfMP_ONLY (pfSPECL [f,g,x,y] seq1) (pfCONJ funeq argeq)
  end

fun shEQ_IMP_RULE_LEFT_err (eq as (_,_,w)) =
  let val (t1,t2) = dest_eq w in 
    pfDISCH t1 (shEQ_MP eq (pfASSUME t1))
  end

fun shEQ_IMP_RULE_RIGHT_err (eq as (_,_,w)) =
  let val (t1,t2) = dest_eq w in
    pfDISCH t2 (shEQ_MP (shSYM eq) (pfASSUME t2))
  end


fun shMK_COMB funeq argeq = 
  catch "shMK_COMB" (shMK_COMB_err funeq) argeq
fun shEQ_IMP_RULE_LEFT eq = 
  catch "shEQ_IMP_RULE_LEFT" shEQ_IMP_RULE_LEFT_err eq
fun shEQ_IMP_RULE_RIGHT eq = 
  catch "shEQ_IMP_RULE_RIGHT" shEQ_IMP_RULE_RIGHT_err eq


end (* struct *)
