structure hfProofTerm :> hfProofTerm =
struct

open HolKernel boolLib aiLib

val ERR = mk_HOL_ERR "hfProofTerm"

(* ------------------------------------------------------------------------
   Proof terms
   ------------------------------------------------------------------------ *)

datatype pf =
    KnO of string
  | PrV of int
  | PrA of pf * pf
  | TmA of pf * term
  | PrL of int * term * pf
  | TmL of term (* var *) * pf
  | ExT of hol_type * hol_type

fun ptts t = 
  if is_var t orelse is_const t 
  then term_to_string t 
  else "(" ^ term_to_string t ^ ")"

fun human_pf pftop = case pftop of
    KnO s => s
  | PrV i => "u" ^ its i
  | PrA (pf1,pf2) => 
    "(" ^ human_pf pf1 ^ " " ^ human_pf pf2 ^ ")"
  | TmA (pf,t) => 
    "(" ^ human_pf pf ^ " " ^ ptts t ^ ")"
  | PrL (i,t,pf) => 
    "(\\" ^ "u" ^ its i ^ ":" ^ ptts t ^ "." ^ human_pf pf ^ ")"
  | TmL (v,pf) => 
    "(\\" ^ ptts v ^ "." ^ human_pf pf ^ ")"
  | ExT (a,b) => "(Ext " ^ type_to_string a ^ " " ^ type_to_string b ^ ")"

(* ------------------------------------------------------------------------
   Proof variable dictionary. Unique proof variables for each hypothesis.
   ------------------------------------------------------------------------ *)

val pfvd_global = ref (dempty Term.compare)

fun gen_pfv t =
  dfind t (!pfvd_global) handle NotFound =>
  let val r = dlength (!pfvd_global) in 
    pfvd_global := dadd t r (!pfvd_global); r 
  end

fun human_pfvl l = String.concatWith ","
  (map (fn (a,b) => "u" ^ its b ^ ":" ^ ptts a) l)

(* ------------------------------------------------------------------------
   Traverse a proof term and apply functions to terms and types 
   ------------------------------------------------------------------------ *)

fun pf_size pftop = case pftop of
    KnO s => 1
  | PrV i => 1
  | PrA (pf1,pf2) => pf_size pf1 + pf_size pf2
  | TmA (pf,t) => pf_size pf + term_size t
  | PrL (i,t,pf) => 1 + term_size t + pf_size pf 
  | TmL (v,pf) => term_size v + pf_size pf
  | ExT (a,b) => 1 + type_size a + type_size b

fun pfsubst_term theta pftop =
  let fun cont x = pfsubst_term theta x in
     case pftop of
      KnO s => pftop
    | PrV i => pftop
    | PrA (pf1,pf2) => PrA (cont pf1, cont pf2)
    | TmA (pf,t) => TmA (cont pf, subst theta t)
    | PrL (i,t,pf) => PrL (i, subst theta t, cont pf) 
    | TmL (v,pf) => 
      let val newtheta = filter (fn x => not (term_eq (#redex x) v)) theta in
        TmL (v, pfsubst_term newtheta pf)
      end
    | ExT (a,b) => pftop
  end

fun pfsubst_type theta pftop =
  let fun cont x = pfsubst_type theta x in
     case pftop of
      KnO s => pftop
    | PrV i => pftop
    | PrA (pf1,pf2) => PrA (cont pf1, cont pf2)
    | TmA (pf,t) => TmA (cont pf, inst theta t)
    | PrL (i,t,pf) => PrL (i, inst theta t, cont pf) 
    | TmL (v,pf) => TmL (inst theta v, cont pf)
    | ExT (a,b) => ExT (type_subst theta a, type_subst theta b)
  end

fun tlist_subset l1 l2 = all (fn x => tmem x l2) l1

fun tm_nfv b vl t =
  if tlist_subset (free_vars t) vl 
  then true
  else (print_endline ((if b then "TmA" else "PrL") ^ term_to_string t); false)

fun no_free_vars vl pftop = 
 let fun cont x = no_free_vars vl x in
   case pftop of
      KnO s => true
    | PrV i => true
    | PrA (pf1,pf2) => cont pf1 andalso cont pf2
    | TmA (pf,t) => cont pf andalso tm_nfv true vl t
    | PrL (i,t,pf) => tm_nfv false vl t andalso cont pf 
    | TmL (v,pf) => no_free_vars (v :: vl) pf
    | ExT (a,b) => true
 end

fun has_free_vars pf = not (no_free_vars [] pf)

fun free_vars_pf pftop1 =
  let 
    fun loop vl pftop2 = case pftop2 of
        KnO s => []
      | PrV i => []
      | PrA (pf1,pf2) => loop vl pf1 @ loop vl pf2
      | TmA (pf,t) => loop vl pf @ free_vars (list_mk_abs (vl,t))
      | PrL (i,t,pf) => free_vars (list_mk_abs (vl,t)) @ loop vl pf
      | TmL (v,pf) => loop (v :: vl) pf
      | ExT (a,b) => []
  in
    mk_term_set (loop [] pftop1)
  end

fun pf_find_term test pftop = 
 let fun cont x = pf_find_term test x in
   case pftop of
      KnO s => false
    | PrV i => false
    | PrA (pf1,pf2) => cont pf1 orelse cont pf2
    | TmA (pf,t) => cont pf orelse test t
    | PrL (i,t,pf) => test t orelse cont pf 
    | TmL (v,pf) => test v orelse cont pf
    | ExT (a,b) => false
 end

fun pf_subterms (pft,d') =
  let 
    val d = ref d'
    fun loop pftop = case pftop of
      KnO s => ()
    | PrV i => ()
    | PrA (pf1,pf2) => (loop pf1; loop pf2)
    | TmA (pf,t) => (d := count_dict (!d) [t]; loop pf)
    | PrL (i,t,pf) => (d := count_dict (!d) [t]; loop pf)
    | TmL (v,pf) => loop pf
    | ExT (a,b) => ()
  in
    loop pft; !d
  end

end
