structure hfResolution :> hfResolution =
struct

open HolKernel boolLib normalForms aiLib smlTimeout hfLib 
  hfProofRule hfProofTerm hfTheory hfLemTheory 

val ERR = mk_HOL_ERR "hfResolution"

(* ------------------------------------------------------------------------
   Resolution rule
   ------------------------------------------------------------------------ *)

val vi = ref 0
fun fresh_v () = 
  let val r = mk_var ("_v" ^ its (!vi), ``:hf$set``) in
    incr vi; r
  end

fun is_rfv x = 
  let val vs = fst (dest_var x) in
    String.isPrefix "_v" vs orelse String.isPrefix "_all" vs
  end

fun all_rfv tm = filter is_rfv (free_vars_lr tm)

fun RENAME_FV cjo th = 
  let 
    val _ = if term_eq (concl th) F then () else raise ERR "RENAME_FV" ""
    val asl = acc_asl cjo th 
    val fvl = if null asl then [] else all_rfv (list_mk_conj asl)
    val sub = map (fn x => {redex = x, residue = fresh_v ()}) fvl
  in
    INST sub th
  end

fun unify_lit lit1 lit2 = 
  let
    val (lit1',lit2') = 
      if is_neg lit1 andalso not (is_neg lit2) 
        then (dest_neg lit1,lit2) 
      else if is_neg lit2 andalso not (is_neg lit1) 
        then (lit1, dest_neg lit2) 
      else raise ERR "resolve" "not opposite"
    val protected = 
      filter (not o is_rfv) (free_vars_lr (mk_conj (lit1,lit2)))
  in 
    Unify.simp_unify_terms protected lit1' lit2'  
  end

val verbose = ref false

fun RESOLVE th1 th2 =
  if can (unify_lit (concl th1)) (concl th2) then
  let 
    val sub = unify_lit (concl th1) (concl th2)
    val (th1',th2') = (INST sub th1, INST sub th2)
    val r = if is_neg (concl th1') 
      then MP th1' th2'
      else MP th2' th1'
  in
    if !verbose then 
     (print_endline (thm_to_string th1);
      print_endline (thm_to_string th2);
      print_endline (thm_to_string r ^ "\n"))
    else ();
    SOME r
  end
  else NONE

(* ------------------------------------------------------------------------
   Moving literals
   ------------------------------------------------------------------------ *)

fun DISCH_EACH cjo th =
  let 
    val asl = acc_asl cjo th 
    fun f x = 
      if is_neg x then CCONTR (dest_neg x) th else NOT_INTRO (DISCH x th)
  in 
    map f asl
  end

fun UNDISCH_LIT th =
  if is_neg (concl th) then UNDISCH th else
    UNDISCH (NOT_INTRO (MP (SPEC (concl th) notnot_intro) th))

(* ------------------------------------------------------------------------
   Resolution search
   ------------------------------------------------------------------------ *)

fun is_proven cjo c = null (acc_asl cjo c)

fun init_search cjo thl =
  let val thl1 = map (RENAME_FV cjo o UNDISCH_LIT) thl in
    List.concat (map (DISCH_EACH cjo) thl1)
  end
    
fun search n cjo mem proc unproc = 
   if n <= 0 then raise ERR "search" "timeout" else
   case unproc of
    [] => raise ERR "search" "saturated"
  | c :: m =>
    if HOLset.member (mem, GEN_ALL (DISCH_ALL c)) 
    then search n cjo mem proc m 
    else
    let
      val cl0 = List.mapPartial (RESOLVE c) (c :: proc)
      val cl1 = map (RENAME_FV cjo) cl0
      val cl2 = List.concat (map (DISCH_EACH cjo) cl1)
      val newmem = HOLset.add (mem, GEN_ALL (DISCH_ALL c)) 
    in
      if exists (is_proven cjo) cl1
      then valOf (List.find (is_proven cjo) cl1)
      else search (n-1) cjo newmem (c :: proc) (m @ cl2)
    end

fun search_pb n cjo thl =
  let val thl1 = init_search cjo thl in
    search n cjo (HOLset.empty thm_compare) [] thl1
  end

end (* struct *)
