(* ========================================================================= *)
(* FILE          : hfScript.sml                                              *)
(* DESCRIPTION   : Axioms the HF theory                                      *)
(* AUTHOR        : (c) Thibault Gauthier, Czech Technical University         *)
(* DATE          : 2020                                                      *)
(* ========================================================================= *)

open HolKernel boolLib aiLib hfFormat hfImportSexp hfImportRaw hfLib

val ERR = mk_HOL_ERR "hfTheory"
(* load "hfImportRaw"; open aiLib hfLib hfFormat hfImportSexp hfImportRaw; *)

val thy = "hf"
val _ = new_theory thy

(* ------------------------------------------------------------------------
   Type
   ------------------------------------------------------------------------ *)

fun create_type s = 
  mk_thy_type {Thy=thy, Tyop=s, Args=[]} handle HOL_ERR _ =>
  (prim_new_type {Thy=thy, Tyop=s} 0;
   mk_thy_type {Thy=thy, Tyop=s, Args=[]})

fun h4_type ty = case ty of
    Base s => create_type s (* mk_thy_type {Thy="hf",Tyop=s,Args=[]} *)
  | TpArr (ty1,ty2) => 
    let val (a,b) = (h4_type ty1, h4_type ty2) in 
      mk_thy_type {Thy="min",Tyop="fun",Args=[a,b]}
    end
  | Prop => bool

(* ------------------------------------------------------------------------
   Constant
   ------------------------------------------------------------------------ *)

fun create_const s =
  let val ty = h4_type (dfind s ctyd) in
    mk_thy_const {Thy=thy, Name=s, Ty=ty} handle HOL_ERR _ =>
    (prim_new_const {Thy=thy, Name=s} ty;
     mk_thy_const {Thy=thy, Name=s, Ty=ty})
  end

fun h4_const s = 
  if dmem s hfcd then dfind s hfcd else create_const s

(* ------------------------------------------------------------------------
   Term
   ------------------------------------------------------------------------ *)

fun h4_term_aux vl hftm = case hftm of
    Var s => (valOf (List.find (fn x => fst (dest_var x) = s) vl)
      handle Option => raise ERR "h4_term" ("Var" ^ s))
  | Const s => h4_const s
  | Ap (t1,t2) => mk_comb (h4_term_aux vl t1, h4_term_aux vl t2)
  | Lam (s,ty,t) => 
    let val v = mk_var (s, h4_type ty) in
      mk_abs (v, h4_term_aux (v :: vl) t)
    end
  | Imp (t1,t2) => mk_imp (h4_term_aux vl t1, h4_term_aux vl t2)
  | All (s,ty,t) => 
    let val v = mk_var (s, h4_type ty) in
      mk_forall (v, h4_term_aux (v :: vl) t)
    end
  | Ex (s,ty,t) =>
    let val v = mk_var (s, h4_type ty) in
      mk_exists (v, h4_term_aux (v :: vl) t)
    end
  | Eq (_,t1,t2) => mk_eq (h4_term_aux vl t1, h4_term_aux vl t2)

fun h4_term hftm = h4_term_aux [] hftm

(* ------------------------------------------------------------------------
   Saving
   ------------------------------------------------------------------------ *)

val _ = prooflog := [];

val _ = 
  let fun f i x = save_thm ("ax" ^ its i, mk_thm ([],h4_term x)) in 
    ignore (mapi f primaxl) 
  end

val _ = 
  let fun f (s,x) = save_thm (s, mk_thm ([],h4_term x)) in 
    ignore (map f pglogicl) 
  end

val _ = export_theory ()
