(* ========================================================================= *)
(* FILE          : proofgold_demo.sml                                        *)
(* DESCRIPTION   : HOL4 as interface to Proofgold HF theory                  *)
(* AUTHOR        : Thibault Gauthier, Czech Technical University             *)
(* DATE          : 2020                                                      *)
(* ========================================================================= *)

load "hfProofRule"; open hfProofRule;
load "hfProofTerm"; open hfProofTerm;
load "hfCheck"; open hfCheck;
load "hfProof"; open hfProof;
load "aiLib"; open aiLib;
load "hfLib"; open hfLib;
load "hfCNF"; open hfCNF;
load "hfExport"; open hfExport;
load "hfTheory"; open hfTheory;
load "hfDep"; open hfDep;
load "hfDev"; open hfDev;
load "mlTacticData"; open mlTacticData;
load "hfProofRule"; open hfProofRule;
load "hfATP"; open hfATP;
load "hfMask"; open hfMask;
load "hfMiner"; open hfMiner;
load "hfResolution"; open hfResolution;
mlibUseful.trace_level := 0; mesonLib.chatting := 0;
show_assums := true;

(* --------------------------------------------------------------------------
   Derivation of HOL4 reflexivity rule from Proofgold kernel rules.
   Reflexivity "t = t" is a kernel rule in HOL4.
   In proofgold, "s = t" stands for "∀Q. Q s t ==> Q t s".
   ------------------------------------------------------------------------- *)

val seq1 = 
  (pfKnO "D", HOLset.empty Term.compare, ``∀x. P (x: hf$set) : bool``);
val seq2 = pfSPEC ``hf$Empty`` seq1;

val t = ``t: hf$set``; 
val a = type_of t;
val qty = list_mk_funtype ([a,a],bool); 
val q = mk_var ("Q",qty);
val qtt = list_mk_comb (q,[t,t]);

val seq1 = pfASSUME qtt;
val seq2 = pfDISCH qtt seq1;
val seq3 = pfGEN q seq2;
val seq4 = pfEQ_FOLD seq3;

(* --------------------------------------------------------------------------
   Manual proof
   ------------------------------------------------------------------------- *)

val bounty = hd (import_terml "zcjproven");
(* you can also try your luck on other elements of this list *) 

(* HOL4 proof *)
val (x0,bod0) = dest_exists bounty;
val (imp0a,imp0b) = dest_conj bod0;
val (x1,bod1) = dest_forall imp0b;
val (imp1a,imp1b) = dest_imp bod1;
val p4 = rand imp0a;

prooflog := []; 
val th1 = AP_THM (AP_THM ax16 p4) p4;
val th2 = CONV_RULE (RAND_CONV LIST_BETA_CONV) th1;
val th3 = ASSUME (mk_comb (``hf$In x2``, p4));
val th4 = GEN ``x2: hf$set`` (DISCH_ALL th3);
val th5 = EQ_MP (SYM th2) th4;
val th6 = REFL ``X1 :hf$set``;
val th7 = DISCH_ALL (ADD_ASSUM imp1a th6);
val th8 = SPEC p4 (GEN ``X0 :hf$set`` th7);
val th9 = CONJ th5 (GEN ``X1 :hf$set`` th8);
val finalthm = EXISTS (bounty,p4) th9;

(* exporting the proofgold proof term *) 
let
  val log = minimize_prooflog ["hf"] [finalthm] (rev (!prooflog))
  val pf = compute_pf (finalthm,log) 
in 
  export_pf "manual_proof.pfg" (finalthm,pf)
end;

(* --------------------------------------------------------------------------
   Automated proof for first-order bounties (all in one)
   ------------------------------------------------------------------------- *)

val bounty = hd (import_terml "zcjproven"); 
(* you can also try your luck on other elements of this list *)

auto_mine_cj "example/automated_proof2.pfg" bounty;


(* --------------------------------------------------------------------------
   Automated proof for first-order bounties (details)
   ------------------------------------------------------------------------- *)

(* holyhammer: translation of 108 axioms of HF theory + 
   ATPs (eprover,vampire,z3) *)
val axl = hh_core bounty;
prooflog := [];
val finalthm = METIS_PROVE axl bounty; 
length (!prooflog);
(* generalizing conjecture *)
val gencj = gen_cj axl bounty;
prooflog := []; 
(* minimizing equality lemmas *)
val eql = create_eql (axl,gencj);
val mineql = minimize_eql (axl,gencj) eql;

(* convert to conjunctive normal form *)
val axlfof = map (GEN_ALL_LR o fofify_ax) axl  
val ncj = mk_neg gencj
val ncjo = SOME ncj
val clausel = CNF_LOOP ncjo (ASSUME ncj :: (axlfof @ mineql));
(* resolution prover *)
val resthm = search_pb 10000 ncjo clausel;
(* eliminate skolem constants + instantiate the general conjecture *)
val finalthm = elim_locdef (bounty,gencj) resthm;

(* exporting the proof *) 
let 
  val log = minimize_prooflog ["hf","hfLem"] [finalthm] (rev (!prooflog));
  val pf = compute_pf (finalthm,log)
in 
  export_pf "example/automated_proof1.pfg" (finalthm,pf)
end;



